/* JDxpc -- DXPC in pure Java
 *
 *  Copyright (C) 2000 ymnk, JCraft, Inc.
 *
 *  Many thanks to 
 *    Brian Pane<brianp@cnet.com> and
 *    Zachary Vonler<lightborn@mail.utexas.edu>.
 *  JDxpc has been based on their awesome works, dxpc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package com.jcraft.jdxpc;

import java.io.*;
import java.net.*;
import java.lang.*;

class ServerChannel extends Channel{
  private boolean firstRequest=true;
  private boolean firstReply=true;

  private int imageByteOrder;
  private int bitmapBitOrder;
  private int scanlineUnit;
  private int scanlinePad;

  InputStream serverInput=null;
  OutputStream serverOutput=null;
  Socket serverSocket=null;

  ServerReadBuffer rBuffer=new ServerReadBuffer();

  ServerCache sCache=new ServerCache();
  SequenceNumQueue sNumQueue=new SequenceNumQueue();
  ClientCache cCache=new ClientCache();

  private final IntCache windowCache=cCache.windowCache;
  private final IntCache drawableCache=cCache.drawableCache;
  private final IntCache gcCache=cCache.gcCache;
  private final CharCache changePropertyFormatCache=cCache.changePropertyFormatCache;
  private final IntCache changePropertyPropertyCache=cCache.changePropertyPropertyCache;
  private final IntCache changePropertyTypeCache=cCache.changePropertyTypeCache;
  private final IntCache changePropertyData32Cache=cCache.changePropertyData32Cache;
  private final TextCompressor changePropertyTextCompressor=cCache.changePropertyTextCompressor;

  private final CharCache[] opcodeCache=cCache.opcodeCache;

  WriteBuffer wBuffer=new WriteBuffer();

  int[] nextOpcode=new int[1];
  int[] start=new int[1];
  int[] length=new int[1];
  byte[] buff;

  int[][] requestData=new int[3][];

  int[] nextSequenceNum=new int[1];
  int[] dummySequenceNum=new int[1];
  int[] dummyOpcode=new int[1];

  ServerChannel(){ this("127.0.0.1", 6000); }
  ServerChannel(String host, int port){
    try	{
      Socket s=new Socket(host, port);
      setSocket(s);
    }
    catch(IOException e) {
      System.out.println("IO Error : " +  e );
    }
    requestData[0]=new int[1];
    requestData[1]=new int[1];
    requestData[2]=new int[1];
  }

  void setSocket(Socket s) {
    serverSocket=s;
    try	{
      serverInput=serverSocket.getInputStream();
      serverOutput=serverSocket.getOutputStream();
      rBuffer.setInputStream(serverInput);
    }
    catch(IOException e) {
      System.out.println("IO Error : " +  e );
    }
  }

  void close(){
    try{ serverSocket.close(); }
    catch(Exception e){}
  }

  public void run(){
    EncodeBuffer eBuffer=new EncodeBuffer();
    while(true){
      try{ rBuffer.doRead(); }
      catch(java.io.IOException e){ 
        eBuffer.reset();
        proxy.proxyDrop(id, eBuffer);
        break;
      }
      doRead(rBuffer, eBuffer);
    }
  }

  private int[] rvalue=new int[1];

  void doRead(ReadBuffer rBuffer, EncodeBuffer eBuffer){
    int s;
    while((buff=rBuffer.getMessage(start, length))!=null){
      eBuffer.reset();
      s=start[0];

      if(firstReply){
	firstReply=false;
	imageByteOrder=buff[s+30];
	bitmapBitOrder=buff[s+31];
	scanlineUnit=buff[s+32];
	scanlinePad=buff[s+32];
	eBuffer.encodeValue(buff[s+0], 8);
	eBuffer.encodeValue(buff[s+1], 8);
	eBuffer.encodeValue(rBuffer.getUINT(2), 16);
	eBuffer.encodeValue(rBuffer.getUINT(4), 16);
	eBuffer.encodeValue(rBuffer.getUINT(6), 16);
	if(ServerCache.lastInitReply.compare(s+8, length[0]-8, buff)){
	  eBuffer.encodeValue(1, 1);
	}
	else{
	  eBuffer.encodeValue(0, 1);
	  for(int i=8; i<length[0]; i++){
	    eBuffer.encodeValue(buff[s+i], 8);
	  }
	}
	proxy.proxyWrite(id, eBuffer);
	continue;
      }

//System.out.println("return "+buff[s+0]);

      byte opcode=buff[s+0];
      if(opcode==1){
	//System.out.println("reply...");
	int sequenceNum=rBuffer.getUINT(2);
	int sequenceNumDiff = sequenceNum-sCache.lastSequenceNum;
	sCache.lastSequenceNum=sequenceNum;

        //System.out.println("seq: "+sequenceNum+" diff: "+sequenceNumDiff);
	//System.out.println("start: "+s+", length: "+length[0]);

	eBuffer.encodeCachedValue(opcode, 8, 
				  sCache.opcodeCache[sCache.lastOpcode]);
	sCache.lastOpcode=(opcode&0xff);
	eBuffer.encodeCachedValue(sequenceNumDiff, 16,
				  sCache.replySequenceNumCache, 7);

	int requestOpcode=256;

	if(sNumQueue.peek(nextSequenceNum, nextOpcode) &&
	   (nextSequenceNum[0]==sequenceNum)){
	  sNumQueue.pop(nextSequenceNum, nextOpcode,
			requestData[0], requestData[1], requestData[2]);
	  requestOpcode=nextOpcode[0];

//System.out.println("  requestOpcode="+requestOpcode);

	  switch(requestOpcode){
	  case 84: // X_AllocColor:
	    {
	      int nextSrc=8;
	      for(int i=0; i<3; i++){
		int colorValue=rBuffer.getUINT(nextSrc);
		nextSrc+=2;
		if(colorValue==requestData[i][0]){
		  eBuffer.encodeValue(1, 1);
		}
		else{
		  eBuffer.encodeValue(0, 1);
		  eBuffer.encodeValue(colorValue-colorValue, 16, 6);
		}
	      }
	      int pixel=rBuffer.getULONG(16);
	      eBuffer.encodeValue(pixel, 32, 9);
	    }
	    break;
	  case 17: //X_GetAtomName:
            {
	      int nameLength = rBuffer.getUINT(4);  //BUG!!
	      //int nameLength = rBuffer.getUINT(8);
	      eBuffer.encodeValue(nameLength, 16, 6);
	      int nextSrc=s+32;
	      cCache.internAtomTextCompressor.reset();
	      for(int i=0; i<nameLength; i++){
		cCache.internAtomTextCompressor.encodeChar(buff[nextSrc++],
							   eBuffer);
	      }
	    }
	    break;
	  case 14: //X_GetGeometry:
	    {
	      eBuffer.encodeCachedValue(buff[s+1], 8, sCache.depthCache);
	      eBuffer.encodeCachedValue(rBuffer.getULONG(8),
					29, sCache.getGeometryRootCache, 9);
	      int nextSrc=12;
	      for(int i=0; i<5; i++){
		eBuffer.encodeCachedValue(rBuffer.getUINT(nextSrc),
					  16, 
					  sCache.getGeometryGeomCache[i], 8);
		nextSrc+=2;
	      }
	    }
	    break;
	  case 43: // GetInputFocus
	    {
	      eBuffer.encodeValue(buff[s+1], 2);
	      eBuffer.encodeCachedValue(rBuffer.getULONG(8),
					29, 
					sCache.getInputFocusWindowCache, 9);
	    }
	    break;
	  case 101: // X_GetKeyboardMapping:
	    {
	      int keysymsPerKeycode = (int)buff[s+1];
	      if(ServerCache.getKeyboardMappingLastMap.compare(s+32,
							       length[0]-32,
							       buff) &&
		 (keysymsPerKeycode==ServerCache.getKeyboardMappingLastKeysymsPerKeycode)){
		eBuffer.encodeValue(1, 1);
		break;
	      }
	      ServerCache.getKeyboardMappingLastKeysymsPerKeycode=(byte)keysymsPerKeycode;
	      eBuffer.encodeValue(0, 1);
	      int numKeycodes=(((length[0]-32)/keysymsPerKeycode)>>>2);
	      eBuffer.encodeValue(numKeycodes, 8);
	      eBuffer.encodeValue(keysymsPerKeycode, 8, 4);
	      int nextSrc=32;
	      byte previous=0;
	      for(int count = numKeycodes * keysymsPerKeycode; count!=0 ; --count){
		int keysym = rBuffer.getULONG(nextSrc);
		nextSrc += 4;
		if(keysym==0/*NoSymbol*/){
		  eBuffer.encodeValue(1, 1);
		}
		else{
		  eBuffer.encodeValue(0, 1);
		  int first3Bytes=(keysym>>>8);
		  eBuffer.encodeCachedValue(first3Bytes, 24,
					    sCache.getKeyboardMappingKeysymCache, 9);
		  byte lastByte=(byte)(keysym/*&0xff*/);
		  eBuffer.encodeCachedValue((byte)(lastByte-previous), 8,
					    sCache.getKeyboardMappingLastByteCache, 5);
		  previous=lastByte;
		}
	      }
	    }
	    break;
	  case 119: // X_GetModifierMapping:
	    {
	      eBuffer.encodeValue((int)buff[s+1], 8);
	      int nextDest=s+32;
	      if(ServerCache.getModifierMappingLastMap.compare(nextDest,
							       length[0]-32,
							       buff)){
		eBuffer.encodeValue(1, 1);
		break;
	      }
	      eBuffer.encodeValue(0, 1);
	      for(int count=length[0]-32;count!=0; count--){
		byte next=buff[nextDest++];
		if(next==0){
		  eBuffer.encodeValue(1, 1);
		}
		else{
		  eBuffer.encodeValue(0, 1);
		  eBuffer.encodeValue(next, 8);
		}
	      }
	    }
	    break;
	  case 20: // GetProperty
	    {
	      int format=(int)buff[s+1];
	      eBuffer.encodeCachedValue((byte)format, 8,
					sCache.getPropertyFormatCache);
	      int numBytes=rBuffer.getULONG(16);
	      eBuffer.encodeValue(numBytes, 32, 9);
	      if(format==16){ 
		numBytes<<=1; 
	      }
	      else if(format==32){
		numBytes<<=2;
	      }
	      eBuffer.encodeCachedValue(rBuffer.getULONG(8),
					29, sCache.getPropertyTypeCache, 9);
	      eBuffer.encodeValue(rBuffer.getULONG(12), 32, 9);
	      int next=s+32;
	      if(format==8) {
		if (requestData[0][0]==23/*XA_RESOURCE_MANAGER*/) {
		  if (ServerCache.xResources.compare(s+32, numBytes, buff)){
		    eBuffer.encodeValue(1, 1);
		    break;
		  }
		  eBuffer.encodeValue(0, 1);
		}
		sCache.getPropertyTextCompressor.reset();
		for(int i=0; i<numBytes; i++){
		  int nextChar=0;
		  sCache.getPropertyTextCompressor.encodeChar((byte)(nextChar=buff[next++]), eBuffer);
		  if(nextChar==10){
		    sCache.getPropertyTextCompressor.reset(nextChar);
		  }
		}
	      }
	      else{
		for(int i=0; i < numBytes; i++){
		  eBuffer.encodeValue(buff[next++], 8);
		}
	      }
	    }
	    break;
	  case 23: //X_GetSelectionOwner:
	    {
	      eBuffer.encodeCachedValue(rBuffer.getULONG(8),
					29, sCache.getSelectionOwnerCache, 9);
	    }
	    break;
	  case 3: // X_GetWindowAttributes:
	    {
	      eBuffer.encodeValue((int)buff[s+1], 2);
	      eBuffer.encodeCachedValue(rBuffer.getULONG(8),
					29, sCache.visualCache, 9);
	      eBuffer.encodeCachedValue(rBuffer.getUINT(12),
					16, sCache.getWindowAttributesClassCache, 3);
	      eBuffer.encodeCachedValue(buff[s+14], 8,
					sCache.getWindowAttributesBitGravityCache);
	      eBuffer.encodeCachedValue(buff[s+15], 8,
					sCache.getWindowAttributesWinGravityCache);
	      eBuffer.encodeCachedValue(rBuffer.getULONG(16),
					32, sCache.getWindowAttributesPlanesCache, 9);
	      eBuffer.encodeCachedValue(rBuffer.getULONG(20),
					32, sCache.getWindowAttributesPixelCache, 9);
	      eBuffer.encodeValue((int)buff[s+24], 1);
	      eBuffer.encodeValue((int)buff[s+25], 1);
	      eBuffer.encodeValue((int)buff[s+26], 2);
	      eBuffer.encodeValue((int)buff[s+27], 1);
	      eBuffer.encodeCachedValue(rBuffer.getULONG(28),
					29, sCache.colormapCache, 9);
	      eBuffer.encodeCachedValue(rBuffer.getULONG(32),
					32, sCache.getWindowAttributesAllEventsCache);
	      eBuffer.encodeCachedValue(rBuffer.getULONG(36),
					32, sCache.getWindowAttributesYourEventsCache);
	      eBuffer.encodeCachedValue(rBuffer.getUINT(40),
					16, sCache.getWindowAttributesDontPropagateCache);
	    }
	    break;
	  case 31: // X_GrabKeyboard:
	  case 26: // X_GrabPointer:
	    {
	      eBuffer.encodeValue((int) buff[s+1], 3);
	    }
	    break;
	  case 16: // InternAtom
	    {
	      eBuffer.encodeValue(rBuffer.getULONG(8), 29, 9);
	    }
	    break;
	  case 99: // ListExtensions
	    {
	      eBuffer.encodeValue(rBuffer.getULONG(4), 32, 8);
	      int numExtensions=(int)buff[s+1];
	      eBuffer.encodeValue(numExtensions, 8);
	      int next=s+32;
	      for(; numExtensions>0; numExtensions--){
		int foo=buff[next++];
		eBuffer.encodeValue(foo, 8);
		if(foo==7 && new String(buff, next, 7).equals("MIT-SHM")){
		  System.arraycopy("NOT-SHM".getBytes(), 0, buff, next, 7);
		}
		for(; foo>0; foo--){
		  eBuffer.encodeValue(buff[next++], 8);
		}
	      }
	    }
	    break;
	  case 49: // X_ListFonts:
	    {
	      eBuffer.encodeValue(rBuffer.getULONG(4), 32, 8);
	      int numFonts=rBuffer.getUINT(8);
	      eBuffer.encodeValue(numFonts, 16, 6);
	      int nextSrc=s+32;
	      for(; numFonts!=0; numFonts--)
	      {
		int length=(int)buff[nextSrc++];
		eBuffer.encodeValue(length, 8);
		sCache.getPropertyTextCompressor.reset();
		for (; length!=0; length--){
		  sCache.getPropertyTextCompressor.encodeChar(buff[nextSrc++], eBuffer);
		}
	      }
	    }
	    break;
	  case 92: // X_LookupColor:
	  case 85: // X_AllocNamedColor:
	    {
	      int nextSrc=8;
	      if(requestOpcode==85/*X_AllocNamedColor*/){
		eBuffer.encodeValue(rBuffer.getULONG(nextSrc), 32, 9);
		nextSrc+=4;
	      }
	      int count=3;
	      do{
		int exactColor=rBuffer.getUINT(nextSrc);
		eBuffer.encodeValue(exactColor, 16, 9);
		int visualColor=rBuffer.getUINT(nextSrc+6)-exactColor;
		eBuffer.encodeValue(visualColor, 16, 5);
		nextSrc+=2;
	      }
	      while(--count!=0);
	    }
	    break;
	  case 97: // QueryBestSize
	    {
	      eBuffer.encodeValue(rBuffer.getUINT(8), 16, 8);
	      eBuffer.encodeValue(rBuffer.getUINT(10), 16, 8);
	    }
	    break;
	  case 91: // X_QueryColors
	    {
	      int numColors=((length[0]-32)>>>3);
	      int nextSrc=s+40;
	      int nextDest=s+38;
	      for(int c=1; c<numColors; c++){
		for(int i=0; i<6; i++)
		  buff[nextDest++]=buff[nextSrc++];
		nextSrc+=2;
	      }
	      int colorsLength=numColors*6;
	      if(sCache.queryColorsLastReply.compare(s+32, colorsLength, buff)){
		eBuffer.encodeValue(1, 1);
	      }
	      else{
		nextSrc=32;
		eBuffer.encodeValue(0, 1);
		eBuffer.encodeValue(numColors, 16, 5);
		for(numColors*=3; numColors!=0; numColors--){
		  eBuffer.encodeValue(rBuffer.getUINT(nextSrc), 16);
		  nextSrc+=2;
		}
	      }
	    }
	    break;
	  case 98: // QueryExtension
	    {
	      if(requestData[0][0]!=0){
		eBuffer.encodeValue(0, 1);
		eBuffer.encodeValue(0, 8);
	      }
	      else{
		eBuffer.encodeValue(buff[s+8], 1);
		eBuffer.encodeValue(buff[s+9], 8);
	      }
	      eBuffer.encodeValue(buff[s+10], 8);
	      eBuffer.encodeValue(buff[s+11], 8);
	    }
	    break;
	  case 47: // X_QueryFont:
	    {
	      int numProperties=rBuffer.getUINT(46);
	      int numCharInfos=rBuffer.getULONG(56);
	      eBuffer.encodeValue(numProperties, 16, 8);
	      eBuffer.encodeValue(numCharInfos, 32, 10);
	      encodeCharInfo(rBuffer, 8, eBuffer);
	      encodeCharInfo(rBuffer, 24, eBuffer);
	      eBuffer.encodeValue(rBuffer.getUINT(40), 16, 9);
	      eBuffer.encodeValue(rBuffer.getUINT(42), 16, 9);
	      eBuffer.encodeValue(rBuffer.getUINT(44), 16, 9);
	      eBuffer.encodeValue((int)buff[s+48], 1);
	      eBuffer.encodeValue((int)buff[s+49], 8);
	      eBuffer.encodeValue((int)buff[s+50], 8);
	      eBuffer.encodeValue((int)buff[s+51], 1);
	      eBuffer.encodeValue(rBuffer.getUINT(52), 16, 9);
	      eBuffer.encodeValue(rBuffer.getUINT(54), 16, 9);
	      int nextSrc=s+60;
	      if(ServerCache.queryFontFontCache.lookup(nextSrc,
						       numProperties*8+numCharInfos * 12,
						       buff, rvalue)){
		eBuffer.encodeValue(1, 1);
		eBuffer.encodeValue(rvalue[0], 4);
		break;
	      }
	      int index=rvalue[0];
	      eBuffer.encodeValue(0, 1);
	      nextSrc=60;
	      for(; numProperties!=0; numProperties--){
		eBuffer.encodeValue(rBuffer.getULONG(nextSrc), 32, 9);
		eBuffer.encodeValue(rBuffer.getULONG(nextSrc+4), 32, 9);
		nextSrc+=8;
	      }
	      for(; numCharInfos!=0; numCharInfos--){
 	        encodeCharInfo(rBuffer, nextSrc, eBuffer);
		nextSrc+=12;
	      }
	    }
	    break;
	  case 38: // X_QueryPointer:
	    {
	      eBuffer.encodeValue((int)buff[s+1], 1);
	      eBuffer.encodeCachedValue(rBuffer.getULONG(8),
					29, sCache.queryPointerRootCache, 9);
	      eBuffer.encodeCachedValue(rBuffer.getULONG(12),
					29, sCache.queryPointerChildCache, 9);
	      int rootX=rBuffer.getUINT(16);
	      int rootY=rBuffer.getUINT(18);
	      int eventX=rBuffer.getUINT(20);
	      int eventY=rBuffer.getUINT(22);
	      eventX-=rootX;
	      eventY-=rootY;
	      eBuffer.encodeCachedValue(rootX-sCache.motionNotifyLastRootX,
					16, sCache.motionNotifyRootXCache, 8);
	      sCache.motionNotifyLastRootX=rootX;
	      eBuffer.encodeCachedValue(rootY-sCache.motionNotifyLastRootY,
					16, sCache.motionNotifyRootYCache, 8);
	      sCache.motionNotifyLastRootY=rootY;
	      eBuffer.encodeCachedValue(eventX, 
					16, sCache.motionNotifyEventXCache, 8);
	      eBuffer.encodeCachedValue(eventY, 
					16, sCache.motionNotifyEventYCache, 8);
	      eBuffer.encodeCachedValue(rBuffer.getUINT(24),
					16, sCache.motionNotifyStateCache);
	    }
	    break;
	  case 15: // X_QueryTree
	    {
	      eBuffer.encodeValue(buff[s+1], 8);
	      eBuffer.encodeValue(rBuffer.getULONG(4), 32);
	      for(int i=8; i<length[0]; i++)
		eBuffer.encodeValue((int)buff[s+i], 8);
	    }
	    break;
	  case 40: // X_TranslateCoords:
	    {
	      eBuffer.encodeValue((int)buff[s+1], 1);
	      eBuffer.encodeCachedValue(rBuffer.getULONG(8),
					29, sCache.translateCoordsChildCache, 9);
	      eBuffer.encodeCachedValue(rBuffer.getUINT(12),
					16, sCache.translateCoordsXCache, 8);
	      eBuffer.encodeCachedValue(rBuffer.getUINT(14),
					16, sCache.translateCoordsYCache, 8);
	    }
	    break;
	  default:
	    //System.out.println("unsupported reply: "+requestOpcode); 
	    break;
	  }
	}
	else{
	  //System.out.println("through...");
	  eBuffer.encodeValue(buff[s+1], 8);
	  eBuffer.encodeValue(rBuffer.getULONG(4), 32);
	  for(int i=8; i<length[0]; i++){
	    eBuffer.encodeValue(buff[s+i], 8);
	  }
	}
	  // reply
	proxy.proxyWrite(id, eBuffer);
	continue;
      }

	int sequenceNum=rBuffer.getUINT(2);
	int sequenceNumDiff=sequenceNum-sCache.lastSequenceNum;
	sCache.lastSequenceNum=sequenceNum;

	eBuffer.encodeCachedValue(opcode, 8,
				  sCache.opcodeCache[(sCache.lastOpcode)]);
	sCache.lastOpcode=(opcode&0xff);
	eBuffer.encodeCachedValue(sequenceNumDiff, 16,
				  sCache.eventSequenceNumCache, 7);

	// check if this is an error that matches a sequence number for
	// which we were expecting a reply

	if (sNumQueue.peek(dummySequenceNum, dummyOpcode) &&
	    ((int)dummySequenceNum[0]==sequenceNum))
	  sNumQueue.pop(dummySequenceNum, dummyOpcode);

	switch(opcode){
	case 0:
	  {
	    byte code = buff[s+1];
	    eBuffer.encodeCachedValue(code, 8, sCache.errorCodeCache);
	    if((code != 11) && (code != 8) && (code != 15) && (code != 1))
	      eBuffer.encodeValue(rBuffer.getULONG(4), 32, 16);
	    if(code >= 18){
	      eBuffer.encodeCachedValue(rBuffer.getUINT(8),
					  16, sCache.errorMinorCache);
	    }
	    eBuffer.encodeCachedValue(buff[s+10], 8, sCache.errorMajorCache);
	    if (code >= 18){
	      int nextSrc = s+11;
	      for(int i = 11; i<32; i++){
		eBuffer.encodeValue(buff[nextSrc++], 8);
	      }
	    }
	  }
	  break;
	case 4: //  ButtonPress
	case 5: // ButtonRelease
	case 2: // KeyPress
	case 3: // KeyRelease
	case 6: // MotionNotify
	case 7: // EnterNotify
	case 8: // LeaveNotify
	  {
	    byte detail = buff[s+1];
	    if (buff[s+0] == 6/*MotionNotify*/){
	      eBuffer.encodeValue((int)detail, 1);
	    }
	    else if ((buff[s+0] == 7/*EnterNotify*/) || 
		     (buff[s+0] == 8/*LeaveNotify*/)){
	      eBuffer.encodeValue((int)detail, 3);
	    }
	    else if (buff[s+0] == 3/*KeyRelease*/){
	      if(detail == sCache.keyPressLastKey){
		eBuffer.encodeValue(1, 1);
	      }
	      else{
		eBuffer.encodeValue(0, 1);
		eBuffer.encodeValue((int) detail, 8);
	      }
	    }
	    else if ((buff[s+0] == 4/*ButtonPress*/) || 
		     (buff[s+0] == 5/*ButtonRelease*/)){
	      eBuffer.encodeCachedValue(detail, 8, sCache.buttonCache);
	    }
	    else{
	      eBuffer.encodeValue((int) detail, 8);
	    }

	    int timestamp = rBuffer.getULONG(4);
	    int timestampDiff = timestamp - sCache.lastTimestamp;
	    sCache.lastTimestamp = timestamp;
	    eBuffer.encodeCachedValue(timestampDiff, 32,
				      sCache.motionNotifyTimestampCache, 9);
	    int skipRest = 0;
	    if(buff[s+0] == 3/*KeyRelease*/){
	      skipRest = 1;
	      for(int i = 8; i < 31; i++){
		if(buff[s+i]!=sCache.keyPressCache[i - 8]){
		  skipRest = 0;
		  break;
		}
	      }
	      eBuffer.encodeValue(skipRest, 1);
	    }
	    if(skipRest==0){
	      int nextSrc=8;
	      for(int i=0; i<3; i++){
		eBuffer.encodeCachedValue(rBuffer.getULONG(nextSrc),
					  29, sCache.motionNotifyWindowCache[i], 6);
		nextSrc+=4;
	      }
	      int rootX=rBuffer.getUINT(20);
	      int rootY=rBuffer.getUINT(22);
	      int eventX=rBuffer.getUINT(24);
	      int eventY=rBuffer.getUINT(26);
	      eventX-=rootX;
	      eventY-=rootY;
	      eBuffer.encodeCachedValue(rootX -
					sCache.motionNotifyLastRootX, 16,
					sCache.motionNotifyRootXCache, 6);
	      sCache.motionNotifyLastRootX = rootX;
	      eBuffer.encodeCachedValue(rootY -
					sCache.motionNotifyLastRootY, 16,
					sCache.motionNotifyRootYCache, 6);
	      sCache.motionNotifyLastRootY = rootY;
	      eBuffer.encodeCachedValue(eventX, 16,
					sCache.motionNotifyEventXCache, 6);
	      eBuffer.encodeCachedValue(eventY, 16,
					sCache.motionNotifyEventYCache, 6);
	      eBuffer.encodeCachedValue(rBuffer.getUINT(28),
					16, sCache.motionNotifyStateCache);
	      if((buff[s+0] == 7/*EnterNotify*/) ||
		 (buff[s+0] == 8/*LeaveNotify*/)){
		eBuffer.encodeValue((int) buff[s+30], 2);
	      }
	      else{
		eBuffer.encodeValue((int) buff[s+30], 1);
	      }
	      if ((buff[s+0] == 7/*EnterNotify*/) || 
		  (buff[s+0] == 8/*LeaveNotify*/)){
		eBuffer.encodeValue((int) buff[s+31], 2);
	      }
	      else if (buff[s+0] == 2/*KeyPress*/){
		sCache.keyPressLastKey = detail;
		for(int i = 8; i < 31; i++){
		  sCache.keyPressCache[i - 8] = buff[s+i];
		}
	      }
	    }
	  }
	  break;
	case 32: // ColormapNotify
          {
            eBuffer.encodeCachedValue(rBuffer.getULONG(4),
                                      29, sCache.colormapNotifyWindowCache, 8);
            eBuffer.encodeCachedValue(rBuffer.getULONG(8),
                                    29, sCache.colormapNotifyColormapCache, 8);
            eBuffer.encodeValue((int) buff[s+12], 1);
            eBuffer.encodeValue((int) buff[s+13], 1);
          }
	  break;
	case 22: // ConfigureNotify
	  {
	    int nextSrc=4;
	    for(int i=0; i<3; i++){
	      eBuffer.encodeCachedValue(rBuffer.getULONG(nextSrc), 29,
					sCache.configureNotifyWindowCache[i], 9);
	      nextSrc+=4;
	    }
	    for(int j=0; j<5; j++){
	      eBuffer.encodeCachedValue(rBuffer.getUINT(nextSrc), 16,
					sCache.configureNotifyGeomCache[j], 8);
	      nextSrc+=2;
	    }
	    eBuffer.encodeValue(buff[s+nextSrc], 1);
	  }
	  break;
        case 16: // CreateNotify
          {
            eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				      29, sCache.createNotifyWindowCache, 9);
            int window = rBuffer.getULONG(8);
            eBuffer.encodeValue(window-sCache.createNotifyLastWindow, 29, 5);
            sCache.createNotifyLastWindow = window;
            int nextSrc=12;
            for(int i = 0; i < 5; i++){
              eBuffer.encodeValue(rBuffer.getUINT(nextSrc), 16, 9);
              nextSrc += 2;
            }
            eBuffer.encodeValue(buff[s+nextSrc], 1);
          }
          break;
	case 12: // Expose
	  {
	    eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				      29, sCache.exposeWindowCache, 9);
	    int nextSrc=8;
	    for(int i = 0; i < 5; i++){
	      eBuffer.encodeCachedValue(rBuffer.getUINT(nextSrc),
					16, sCache.exposeGeomCache[i], 6);
	      nextSrc+=2;
	    }
	  }
	  break;
	case  9: // FocusIn:
	case 10: // FocusOut:
	  {
	    eBuffer.encodeValue((int)buff[s+1], 3);
	    eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				      29, sCache.focusInWindowCache, 9);
	    eBuffer.encodeValue((int)buff[s+8], 2);
	  }
	  break;
        case 11: // KeymapNotify
          {
	    if(ServerCache.lastKeymap.compare(s+1, 31, buff)){
              eBuffer.encodeValue(1, 1);
	    }
            else{
              eBuffer.encodeValue(0, 1);
              int nextSrc=s+1;
              for(int i=1; i<32; i++)
                eBuffer.encodeValue((int)buff[nextSrc++], 8);
	    }
	  }
          break;
	case 19: // MapNotify
	case 18: // UnmapNotify
	case 17: // DestroyNotify
	  {
	    eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				      29, sCache.mapNotifyEventCache, 9);
	    eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				      29, sCache.mapNotifyWindowCache, 9);
	    if ((opcode == 19/*MapNotify*/) || (opcode == 18/*UnmapNotify*/))
	      eBuffer.encodeValue((int) buff[s+12], 1);
	  }
	  break;
	case 14: // NoExpose:
	  {
	    eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				      29, sCache.noExposeDrawableCache, 9);
	    eBuffer.encodeCachedValue(rBuffer.getUINT(8), 16,
				      sCache.noExposeMinorCache);
	    eBuffer.encodeCachedValue(buff[s+10], 8, sCache.noExposeMajorCache);
	  }
	  break;
	case 28: // PropertyNotify:
	  {
	    eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				      29, sCache.propertyNotifyWindowCache, 9);
	    eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				      29, sCache.propertyNotifyAtomCache, 9);
	    int timestamp = rBuffer.getULONG(12);
	    int timestampDiff = timestamp-sCache.lastTimestamp;
	    sCache.lastTimestamp = timestamp;
	    eBuffer.encodeValue(timestampDiff, 32, 9);
	    eBuffer.encodeValue((int)buff[s+16], 1);
	  }
	  break;
	case 21: // ReparentNotify
	  {
	    int nextSrc=4;
	    for(int i = 0; i < 3; i++){
	      eBuffer.encodeCachedValue(rBuffer.getULONG(nextSrc),
					29, 
					sCache.reparentNotifyWindowCache, 9);
	      nextSrc+=4;
	    }
	    eBuffer.encodeValue(rBuffer.getUINT(nextSrc), 16, 6);
	    eBuffer.encodeValue(rBuffer.getUINT(nextSrc+2), 16, 6);
	    eBuffer.encodeValue((int)buff[s+20], 1);
	  }
	  break;
	case 29: // SelectionClear:
	  {
	    int timestamp = rBuffer.getULONG(4);
	    int timestampDiff = timestamp-sCache.lastTimestamp;
	    sCache.lastTimestamp = timestamp;
	    eBuffer.encodeValue(timestampDiff, 32, 9);
	    eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				      29, sCache.selectionClearWindowCache, 9);
	    eBuffer.encodeCachedValue(rBuffer.getULONG(12),
				      29, sCache.selectionClearAtomCache, 9);
	  }
	  break;
	case 30: // SelectionRequest:
	  {
	    int timestamp = rBuffer.getULONG(4);
	    int timestampDiff =timestamp-sCache.lastTimestamp;
	    sCache.lastTimestamp = timestamp;
	    eBuffer.encodeValue(timestampDiff, 32, 9);
	    eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				      29, sCache.selectionClearWindowCache, 9);
	    eBuffer.encodeCachedValue(rBuffer.getULONG(12),
				      29, sCache.selectionClearWindowCache, 9);
	    eBuffer.encodeCachedValue(rBuffer.getULONG(16),
				      29, sCache.selectionClearAtomCache, 9);
	    eBuffer.encodeCachedValue(rBuffer.getULONG(20),
				      29, sCache.selectionClearAtomCache, 9);
	    eBuffer.encodeCachedValue(rBuffer.getULONG(24),
				      29, sCache.selectionClearAtomCache, 9);
	  }
	  break;
        case 15: // VisibilityNotify:
          {
            eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				      29, sCache.visibilityNotifyWindowCache, 9);
            eBuffer.encodeValue((int) buff[s+8], 2);
          }
          break;
	default:
	  {
	    if((opcode&0xff)<34 
               && (opcode&0xff)!=31
               && (opcode&0xff)!=13
	       ){
	      //System.out.println("not supported: "+((opcode)&0xff));
	    }
	    eBuffer.encodeValue(buff[s+1], 8);
	    for(int i=4; i<length[0]; i++)
	      eBuffer.encodeValue((int) buff[s+i], 8);
	  }
	}
	proxy.proxyWrite(id, eBuffer);
      }
  }

  void encodeCharInfo(ReadBuffer rBuffer, int start, EncodeBuffer eBuffer){
    int value=rBuffer.getUINT(start)|
              (rBuffer.getUINT(start+10)<<16);
    //System.out.println("encodeCharInfo: "+Integer.toHexString(value));
    eBuffer.encodeCachedValue(value, 32,
			      sCache.queryFontCharInfoCache[0], 6);
    start+=2;
    for(int i=1; i<5; i++){
      value=rBuffer.getUINT(start);
      start+=2;
      eBuffer.encodeCachedValue(value, 16, sCache.queryFontCharInfoCache[i], 6);
    }
  }

  void putByte(byte[] array, int begin, int length) {
    try { serverOutput.write(array, begin, length); }
    catch (IOException e) {
    }
  }

  private int[] wvalue=new int[1];
  private byte[] opcode=new byte[1];
  private byte[] cValue=new byte[1];
  DecodeBuffer dBuffer=new DecodeBuffer(null, 0, 0);

  void doWrite(byte[] buff, int start, int length){
    byte[] message=null;
    int messageStart=0;
    int messageLength=0;

    dBuffer.reset(buff, start, length);

    if(firstRequest){
      wBuffer.reset();
      firstRequest=false;
      messageStart=wBuffer.addMessage(length);
      message=wBuffer.getData();
      int nextDest=messageStart;
      for(int i=0; i<length; i++){
	dBuffer.decodeValue(wvalue, 8);
	message[nextDest++]=(byte)wvalue[0]; 
      }
      if(message[messageStart]==0x42){
	setBigEndian(true);
      }
      else{
	setBigEndian(false);
      }
      putByte(wBuffer.getData(), 0, wBuffer.getLength());
      return;
    }

    while(dBuffer.decodeCachedValue(opcode, 8,
				    opcodeCache[cCache.lastOpcode], 8, true)){

//System.out.println("opcode="+opcode[0]);
      wBuffer.reset();
      cCache.lastOpcode=(opcode[0]&0xff);
      cCache.lastRequestSequenceNum++;
      if((cCache.lastRequestSequenceNum&~0xffff)!=0){
        cCache.lastRequestSequenceNum&=0xffff;
      }
      int iopcode=(opcode[0]&0xff);
      switch(iopcode){
      case 84: // X_AllocColor:
	{
	  messageLength=16;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  dBuffer.decodeCachedValue(wvalue, 29, cCache.colormapCache, 9);
	  wBuffer.putULONG(wvalue[0], 4);
	  int nextDest=/*outputMessage+*/8;
	  int[] colorData=new int[3];
	  for(int i=0; i<3; i++){
	    dBuffer.decodeCachedValue(wvalue, 16,
				      cCache.allocColorRGBCache[i], 4);
  	    wBuffer.putUINT(wvalue[0], nextDest);
	    colorData[i] = wvalue[0];
	    nextDest+=2;
	  }
	  sNumQueue.push(cCache.lastRequestSequenceNum, iopcode,
			 colorData[0], colorData[1], colorData[2]);
	}
	break;
      case 18: // X_ChangeProperty
	{
	  dBuffer.decodeCachedValue(cValue, 8,
				    changePropertyFormatCache);
	  byte format=cValue[0];
	  dBuffer.decodeValue(wvalue, 32, 6);
	  int dataLength=wvalue[0];

	  messageLength=24+(dataLength*(format>>>3)+3)/4*4;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeValue(wvalue, 2);
	  message[messageStart+1] = (byte)wvalue[0];
	  dBuffer.decodeCachedValue(wvalue, 29, windowCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 29,
				    changePropertyPropertyCache, 9);
	  wBuffer.putULONG(wvalue[0], 8);
	  dBuffer.decodeCachedValue(wvalue, 29,
				    changePropertyTypeCache, 9);
          wBuffer.putULONG(wvalue[0], 12);
	  message[messageStart+16]=format;
          wBuffer.putULONG(dataLength, 20);
//	  int nextDest=messageStart+24;
          int nextDest=24;
	  if(format==8){
	    nextDest+=messageStart;
	    changePropertyTextCompressor.reset();
	    for(int i=0; i<dataLength; i++){
	      message[nextDest++]=
		changePropertyTextCompressor.decodeChar(dBuffer);
	    }
	  }
	  else if(format==32){
	    for(int i=0; i<dataLength; i++){
	      dBuffer.decodeCachedValue(wvalue, 32, changePropertyData32Cache);
	      wBuffer.putULONG(wvalue[0], nextDest);
	      nextDest+=4;
	    }
	  }
	  else{
	    for(int i=0; i<dataLength; i++){
	      dBuffer.decodeValue(wvalue, 16);
	      wBuffer.putUINT(wvalue[0], nextDest);
	      nextDest+=2;
	    }
	  }
	}
	break;
      case 2: // X_ChangeWindowAttributes
	{
	  dBuffer.decodeValue(wvalue, 4);
	  int numAttrs=wvalue[0];

	  messageLength=12+(numAttrs<<2);
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeCachedValue(wvalue, 29, windowCache, 9);
	  wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 15,
				    cCache.createWindowBitmaskCache);
	  int bitmask=wvalue[0];
	  wBuffer.putULONG(bitmask, 8);
	  int nextDest=/*messageStart+*/12;
	  int mask=0x1;
	  for(int i=0; i<15; i++){
	    if((bitmask&mask)!=0){
	      dBuffer.decodeCachedValue(wvalue, 32,
					cCache.createWindowAttrCache[i]);
	      wBuffer.putULONG(wvalue[0], nextDest);
	      nextDest+=4;
	    }
	    mask<<=1;
	  }
	}
	break;
      case 61: // X_ClearArea:
	{
	  messageLength=16;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeValue(wvalue, 1);
	  message[messageStart+1]=(byte)wvalue[0];
	  dBuffer.decodeCachedValue(wvalue, 29, windowCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  int nextDest=/*messageStart+*/8;
	  for(int i=0; i<4; i++){
	    dBuffer.decodeCachedValue(wvalue, 16, cCache.clearAreaGeomCache[i], 8);
            wBuffer.putUINT(wvalue[0], nextDest);
	    nextDest+=2;
	  }
	}
	break;
      case 46: // X_CloseFont:
	{
	  messageLength=8;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  dBuffer.decodeValue(wvalue, 29, 5);
	  cCache.lastFont+=wvalue[0];
	  cCache.lastFont&=0x1fffffff;
	  wBuffer.putULONG(cCache.lastFont, 4);
	}
	break;
      case 12: // X_ConfigureWindow
	{
	  messageLength=12;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  wBuffer.registerPointer(messageStart);
	  dBuffer.decodeCachedValue(wvalue, 29, windowCache, 9);
	  wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 7, cCache.configureWindowBitmaskCache);
	  int bitmask=wvalue[0];
	  wBuffer.putUINT(bitmask, 8);
	  int mask=0x1;
	  for(int i=0; i<7; i++){
	    if((bitmask & mask)!=0){
	      wBuffer.addMessage(4);
              message=wBuffer.getData();
              messageLength+=4;
	      dBuffer.decodeCachedValue(wvalue,
					Constants.CONFIGUREWINDOW_FIELD_WIDTH[i],
					cCache.configureWindowAttrCache[i], 8);
	      wBuffer.putULONG(wvalue[0], 0);
	    }
	    mask<<=1;
	  }
	  wBuffer.unregisterPointer();
	}
	break;
      case 24: // X_ConvertSelection:
	{
	  messageLength=24;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeCachedValue(wvalue, 29,
				    cCache.convertSelectionRequestorCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  int nextDest=8;
	  for(int i=0; i<3; i++){
	    dBuffer.decodeCachedValue(wvalue, 29,
				      cCache.convertSelectionAtomCache[i], 9);
	    wBuffer.putULONG(wvalue[0], nextDest);
	    nextDest+=4;
	  }
	  dBuffer.decodeValue(wvalue, 32, 4);
	  cCache.convertSelectionLastTimestamp+=wvalue[0];
	  wBuffer.putULONG(cCache.convertSelectionLastTimestamp, nextDest);
	}
	break;
      case 62: // X_CopyArea:
	{
	  messageLength=28;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeCachedValue(wvalue, 29, drawableCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 29, drawableCache, 9);
          wBuffer.putULONG(wvalue[0], 8);
	  dBuffer.decodeCachedValue(wvalue, 29, gcCache, 9);
          wBuffer.putULONG(wvalue[0], 12);
	  int nextDest=/* messageStart+*/16;
	  for(int i = 0; i < 6; i++){
	    dBuffer.decodeCachedValue(wvalue, 16,
				      cCache.copyAreaGeomCache[i], 8);
            wBuffer.putUINT(wvalue[0], nextDest);
	    nextDest+=2;
	  }
	}
	break;
      case 57: // X_CopyGC:
	{
	  messageLength=16;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeCachedValue(wvalue, 29, gcCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 29, gcCache, 9);
          wBuffer.putULONG(wvalue[0], 8);
	  dBuffer.decodeCachedValue(wvalue, 23,
				    cCache.createGCBitmaskCache);
          wBuffer.putULONG(wvalue[0], 12);
	}
	break;
      case 63: // X_CopyPlane:
	{
	  messageLength = 32;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeCachedValue(wvalue, 29, drawableCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 29, drawableCache, 9);
          wBuffer.putULONG(wvalue[0], 8);
	  dBuffer.decodeCachedValue(wvalue, 29, gcCache, 9);
          wBuffer.putULONG(wvalue[0], 12);
	  int nextDest=/*messageStart+*/16;
	  for(int i=0; i<6; i++){
	    dBuffer.decodeCachedValue(wvalue, 16, cCache.copyPlaneGeomCache[i], 8);
            wBuffer.putUINT(wvalue[0], nextDest);
	    nextDest+=2;
	  }
	  dBuffer.decodeCachedValue(wvalue, 32, cCache.copyPlaneBitPlaneCache, 10);
          wBuffer.putULONG(wvalue[0], 28);
	}
	break;
      case 55: // X_CreateGC:
      case 56: // X_ChangeGC:
	{
	  messageLength=12;
	  if(iopcode==55/*X_CreateGC*/)
	    messageLength+=4;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  wBuffer.registerPointer(messageStart);
	  dBuffer.decodeCachedValue(wvalue, 29, gcCache, 9);
	  wBuffer.putULONG(wvalue[0], 4);
	  int offset=8;
	  if(iopcode==55/*X_CreateGC*/){             
	    dBuffer.decodeCachedValue(wvalue, 29, drawableCache, 9);
	    wBuffer.putULONG(wvalue[0], offset);
	    offset+=4;
	  }
	  dBuffer.decodeCachedValue(wvalue, 23, cCache.createGCBitmaskCache);
	  int bitmask=wvalue[0];
	  wBuffer.putULONG(bitmask, offset);
	  int mask=1;
	  for(int i=0; i<23; i++){
	    if((bitmask&mask)!=0){
//              int nextDest=wBuffer.addMessage(4);
	      wBuffer.addMessage(4);
	      message=wBuffer.getData();
	      messageLength+=4;
	      int fieldWidth=Constants.CREATEGC_FIELD_WIDTH[i];
	      if (fieldWidth<=4){
		dBuffer.decodeValue(wvalue, fieldWidth);
	      }
	      else{
		dBuffer.decodeCachedValue(wvalue, fieldWidth,
					  cCache.createGCAttrCache[i]);
	      }
	      wBuffer.putULONG(wvalue[0], 0);
	    }
	    mask<<=1;
	  }
	  wBuffer.unregisterPointer();
	}
	break;
      case 53: // X_CreatePixmap
	{
	  messageLength = 16;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeCachedValue(cValue, 8, cCache.depthCache);
	  message[messageStart+1]=cValue[0];
	  dBuffer.decodeValue(wvalue, 1);
	  if(wvalue[0]==0){
	    dBuffer.decodeValue(wvalue, 29, 4);
	    cCache.createPixmapLastPixmap+=wvalue[0];
	    cCache.createPixmapLastPixmap&=0x1fffffff;
	  }
          wBuffer.putULONG(cCache.createPixmapLastPixmap, 4);
	  dBuffer.decodeCachedValue(wvalue, 29, drawableCache, 9);
          wBuffer.putULONG(wvalue[0], 8);
	  dBuffer.decodeCachedValue(wvalue, 16,
				    cCache.createPixmapXCache, 8);
          wBuffer.putUINT(wvalue[0], 12);
	  dBuffer.decodeCachedValue(wvalue, 16,
				    cCache.createPixmapYCache, 8);
          wBuffer.putUINT(wvalue[0], 14);
	}
	break;
      case 1: // X_CreateWindow
	{
	  messageLength=32;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  wBuffer.registerPointer(messageStart);
	  dBuffer.decodeCachedValue(cValue, 8, cCache.depthCache);
	  message[messageStart+1]=cValue[0];
	  dBuffer.decodeCachedValue(wvalue, 29, windowCache, 9);
	  wBuffer.putULONG(wvalue[0], 8);
	  dBuffer.decodeCachedValue(wvalue, 29, windowCache, 9);
	  wBuffer.putULONG(wvalue[0], 4);
	  int nextDest=/*messageStart+*/12;
	  for(int i=0; i<6; i++){
	    dBuffer.decodeCachedValue(wvalue, 16,
				      cCache.createWindowGeomCache[i], 8);
	    wBuffer.putULONG(wvalue[0], nextDest);
	    nextDest+=2;
	  }
	  dBuffer.decodeCachedValue(wvalue, 29, cCache.visualCache);
	  wBuffer.putULONG(wvalue[0], 24);
	  dBuffer.decodeCachedValue(wvalue, 15,
				    cCache.createWindowBitmaskCache);
	  int bitmask=wvalue[0];
	  wBuffer.putULONG(bitmask, 28);
	  int mask=0x1;
	  for(int i=0; i<15; i++){
	    if((bitmask&mask)!=0){
	      /*nextDest=*/wBuffer.addMessage(4);
	      messageLength+=4;
	      message=wBuffer.getData();
	      dBuffer.decodeCachedValue(wvalue, 32,
					cCache.createWindowAttrCache[i]);
	      wBuffer.putULONG(wvalue[0], 0);
	    }
	    mask<<=1;
	  }
	  wBuffer.unregisterPointer();
	}
	break;
      case 19: // X_DeleteProperty:
	{
	  messageLength=12;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeCachedValue(wvalue, 29, windowCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeValue(wvalue, 29, 9);
          wBuffer.putULONG(wvalue[0], 8);
	}
	break;
      case 69: // X_FillPoly
	{
	  dBuffer.decodeCachedValue(wvalue, 14,
				    cCache.fillPolyNumPointsCache, 4);
	  int numPoints=wvalue[0];

	  messageLength=16+(numPoints << 2);
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeCachedValue(wvalue, 29,
				    drawableCache, 9);
	  wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 29, gcCache, 9);
	  wBuffer.putULONG(wvalue[0], 8);
	  dBuffer.decodeValue(wvalue, 2);
	  message[messageStart+12] = (byte)wvalue[0];
	  dBuffer.decodeValue(wvalue, 1);
	  int relativeCoordMode=wvalue[0];
	  message[messageStart+13] = (byte)relativeCoordMode;
	  int nextDest =/*messageStart+*/16;
	  int pointIndex = 0;
	  for(int i = 0; i < numPoints; i++){
	    if(relativeCoordMode!=0){
	      dBuffer.decodeCachedValue(wvalue, 16,
					cCache.fillPolyXRelCache[pointIndex], 8);
	      wBuffer.putUINT(wvalue[0], nextDest);
	      nextDest+=2;
	      dBuffer.decodeCachedValue(wvalue, 16,
					cCache.fillPolyYRelCache[pointIndex], 8);
	      wBuffer.putUINT(wvalue[0], nextDest);
	      nextDest+=2;
	    }
	    else{
	      int x, y;
	      dBuffer.decodeValue(wvalue, 1);
	      if(wvalue[0]!=0){
		dBuffer.decodeValue(wvalue, 3);
		x=cCache.fillPolyRecentX[wvalue[0]];
		y=cCache.fillPolyRecentY[wvalue[0]];
	      }
	      else{
		dBuffer.decodeCachedValue(wvalue, 16,
					  cCache.fillPolyXAbsCache[pointIndex], 8);
		x=wvalue[0];
		dBuffer.decodeCachedValue(wvalue, 16,
					  cCache.fillPolyYAbsCache[pointIndex], 8);
		y=wvalue[0];
		cCache.fillPolyRecentX[cCache.fillPolyIndex]=x;
		cCache.fillPolyRecentY[cCache.fillPolyIndex]=y;
		cCache.fillPolyIndex++;
		if(cCache.fillPolyIndex==8)
		  cCache.fillPolyIndex=0;
	      }
	      wBuffer.putUINT(x, nextDest);
	      nextDest+=2;
	      wBuffer.putUINT(y, nextDest);
	      nextDest+=2;
	    }
	    if(pointIndex+1<Constants.FILL_POLY_MAX_POINTS)
	      pointIndex++;
	  }
	}
	break;
      case 88: // X_FreeColors:
	{
	  dBuffer.decodeValue(wvalue, 16, 4);
	  int numPixels=wvalue[0];

	  messageLength=12+(numPixels<<2);
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeCachedValue(wvalue, 29, cCache.colormapCache, 9);
	  wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeValue(wvalue, 32, 4);
	  wBuffer.putULONG(wvalue[0], 8);
	  int nextDest=12;
	  while(numPixels>0){
	    dBuffer.decodeValue(wvalue, 32, 8);
	    wBuffer.putULONG(wvalue[0], nextDest);
	    nextDest+=4;
	    numPixels--;
	  }
	}
	break;
      case 95: // X_FreeCursor:
	{
	  messageLength=8;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeCachedValue(wvalue, 29,
				    cCache.cursorCache, 9);
	  wBuffer.putULONG(wvalue[0], 4);
	}
	break;
      case 60: // FreeGC
	{
	  messageLength=8;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeCachedValue(wvalue, 29, gcCache, 9);
	  wBuffer.putULONG(wvalue[0], 4);
	}
	break;
      case 54: // X_FreePixmap:
	{
	  messageLength=8;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeValue(wvalue, 1);
	  if(wvalue[0]==0){
	    dBuffer.decodeValue(wvalue, 29, 4);
	    cCache.createPixmapLastPixmap+=wvalue[0];
	    cCache.createPixmapLastPixmap&=0x1fffffff;
	  }
	  wBuffer.putULONG(cCache.createPixmapLastPixmap, 4);
	}
	break;
      case 17: // X_GetAtomName:
	{
	  messageLength=8;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeValue(wvalue, 29, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  sNumQueue.push(cCache.lastRequestSequenceNum, iopcode);
	}
	break;
      case 14: //X_GetGeometry:
	{
	  messageLength=8;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  dBuffer.decodeCachedValue(wvalue, 29, drawableCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  sNumQueue.push(cCache.lastRequestSequenceNum, iopcode);
	}
	break;
      case 43:  // GetInputFocus
      case 119: // GetModifierMapping
	{
	  messageLength=4;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  sNumQueue.push(cCache.lastRequestSequenceNum, iopcode,
			 /*hideExtension*/0);
	}
	break;
      case 101: //X_GetKeyboardMapping:
	{
	  messageLength=8;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeValue(wvalue, 8);
	  message[messageStart+4]=(byte)wvalue[0];
	  dBuffer.decodeValue(wvalue, 8);
	  message[messageStart+5]=(byte)wvalue[0];
	  sNumQueue.push(cCache.lastRequestSequenceNum, iopcode);
	}
	break;
      case 20: // GetProperty
	{
	  messageLength=24;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeValue(wvalue, 1);
	  message[messageStart+1]=(byte)wvalue[0];
	  dBuffer.decodeCachedValue(wvalue, 29,
				    windowCache, 9);
	  wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeValue(wvalue, 29, 9);
	  int property=wvalue[0];
	  wBuffer.putULONG(property, 8);
	  dBuffer.decodeValue(wvalue, 29, 9);
	  wBuffer.putULONG(wvalue[0], 12);
	  dBuffer.decodeValue(wvalue, 32, 2);
	  wBuffer.putULONG(wvalue[0], 16);
	  dBuffer.decodeValue(wvalue, 32, 8);
	  wBuffer.putULONG(wvalue[0], 20);
	  sNumQueue.push(cCache.lastRequestSequenceNum, iopcode, property);
	}
	break;

      case 23: // X_GetSelectionOwner:
	{
	  messageLength=8;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  dBuffer.decodeCachedValue(wvalue, 29,
				    cCache.getSelectionOwnerSelectionCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  sNumQueue.push(cCache.lastRequestSequenceNum, iopcode);
	}
	break;
      case 28: // X_GrabButton:
      case 26: // X_GrabPointer:
	{
	  messageLength = 24;
	  messageStart = wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  dBuffer.decodeValue(wvalue, 1);
	  message[messageStart+1] = (byte)wvalue[0];
	  dBuffer.decodeCachedValue(wvalue, 29, windowCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 16,cCache.grabButtonEventMaskCache);
          wBuffer.putUINT(wvalue[0], 8);
	  dBuffer.decodeValue(wvalue, 1);
	  message[messageStart+10]=(byte)wvalue[0];
	  dBuffer.decodeValue(wvalue, 1);
	  message[messageStart+11] =(byte)wvalue[0];
	  dBuffer.decodeCachedValue(wvalue, 29,
				    cCache.grabButtonConfineCache, 9);
          wBuffer.putULONG(wvalue[0], 12);
	  dBuffer.decodeCachedValue(wvalue, 29, cCache.cursorCache, 9);
          wBuffer.putULONG(wvalue[0], 16);
	  if(iopcode==28/*X_GrabButton*/){
	    dBuffer.decodeCachedValue(cValue, 8, cCache.grabButtonButtonCache);
	    message[messageStart+20] = cValue[0];
	    dBuffer.decodeCachedValue(wvalue, 16, cCache.grabButtonModifierCache);
            wBuffer.putUINT(wvalue[0], 22);
	  }
	  else {
	    dBuffer.decodeValue(wvalue, 32, 4);
	    cCache.grabKeyboardLastTimestamp+=wvalue[0];
            wBuffer.putULONG(cCache.grabKeyboardLastTimestamp, 20);
	    sNumQueue.push(cCache.lastRequestSequenceNum, opcode[0]&0xff);
	  }
	}
	break;
      case 31: // X_GrabKeyboard:
	{
	  messageLength=16;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeValue(wvalue, 1);
	  message[messageStart+1]=(byte)wvalue[0];
	  dBuffer.decodeCachedValue(wvalue, 29, windowCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeValue(wvalue, 32, 4);
	  cCache.grabKeyboardLastTimestamp+=wvalue[0];
          wBuffer.putULONG(cCache.grabKeyboardLastTimestamp, 8);
	  dBuffer.decodeValue(wvalue, 1);
	  message[messageStart+12]=(byte)wvalue[0];
	  dBuffer.decodeValue(wvalue, 1);
	  message[messageStart+13]=(byte)wvalue[0];
	  sNumQueue.push(cCache.lastRequestSequenceNum, iopcode);
	}
	break;
      case 36: // X_GrabServer:
      case 37: // X_UngrabServer:
      case 127: // X_NoOperation:
	{
	  messageLength=4;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	}
	break;
      case 76: // X_ImageText8:
	{
	  dBuffer.decodeValue(wvalue, 8);
	  int textLength=wvalue[0];
	  messageLength=16+(textLength+3)/4*4;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  message[messageStart+1] = (byte)textLength;
	  dBuffer.decodeCachedValue(wvalue, 29, drawableCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 29, gcCache, 9);
          wBuffer.putULONG(wvalue[0], 8);
	  dBuffer.decodeCachedValue(wvalue, 16, cCache.imageText8CacheX, 8);
	  cCache.imageText8LastX+=wvalue[0];
	  cCache.imageText8LastX&=0xffff;
          wBuffer.putUINT(cCache.imageText8LastX, 12);
	  dBuffer.decodeCachedValue(wvalue, 16, cCache.imageText8CacheY, 8);
	  cCache.imageText8LastY+=wvalue[0];
	  cCache.imageText8LastY&=0xffff;
          wBuffer.putUINT(cCache.imageText8LastY, 14);
	  int nextDest=messageStart+16;
	  cCache.imageText8TextCompressor.reset();
	  while(textLength!=0){
	    message[nextDest++]=
	      cCache.imageText8TextCompressor.decodeChar(dBuffer);
	    textLength--;
	  }
	}
	break;
      case 16: // InternAtom
	{
	  dBuffer.decodeValue(wvalue, 16, 6);
	  int nameLength=wvalue[0];
	  messageLength=(nameLength+3)/4*4 + 8;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  wBuffer.putUINT(nameLength, 4);
	  dBuffer.decodeValue(wvalue, 1);
	  message[messageStart+1]=(byte)wvalue[0];
	  int nextDest=messageStart+8;
	  cCache.internAtomTextCompressor.reset();
	  for (int i=0; i < nameLength; i++){
	    message[nextDest++]=
	      cCache.internAtomTextCompressor.decodeChar(dBuffer);
	  }
	  sNumQueue.push(cCache.lastRequestSequenceNum, iopcode);
	}
	break;
      case 99: // ListExtensios
	{
	  messageLength=4;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  sNumQueue.push(cCache.lastRequestSequenceNum, iopcode);
	}
	break;
      case 49: // X_ListFonts:
	{
	  dBuffer.decodeValue(wvalue, 16, 6);
	  int textLength=wvalue[0];

	  messageLength=8+(textLength+3)/4*4;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

          wBuffer.putUINT(textLength, 6);
	  dBuffer.decodeValue(wvalue, 16, 6);
          wBuffer.putUINT(wvalue[0], 4);
	  int nextDest = messageStart+8;
	  cCache.polyText8TextCompressor.reset();
	  for(int i=0; i<textLength; i++){
	    message[nextDest++] =
	      cCache.polyText8TextCompressor.decodeChar(dBuffer);
	  }
	  sNumQueue.push(cCache.lastRequestSequenceNum, iopcode);
	}
	break;
      case 92: // X_LookupColor:
      case 85: // X_AllocNamedColor:
	{
	  dBuffer.decodeValue(wvalue, 16, 6);
	  int textLength=wvalue[0];
	  messageLength=12+(textLength+3)/4*4;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  dBuffer.decodeCachedValue(wvalue, 29, cCache.colormapCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
          wBuffer.putUINT(textLength, 8);
	  int nextDest=messageStart+12;
	  cCache.polyText8TextCompressor.reset();
	  for(int i=0; i<textLength; i++){
	    message[nextDest++] =
	      cCache.polyText8TextCompressor.decodeChar(dBuffer);
	  }
	  sNumQueue.push(cCache.lastRequestSequenceNum, iopcode);
	}
	break;
      case 8:   // X_MapWindow:
      case 10:  // X_UnmapWindow:
      case 9:  // X_MapSubwindows:
      case 3:   // X_GetWindowAttributes:
      case 4:   // X_DestroyWindow:
      case 5:   // X_DestroySubwindows:
      case 38:  // X_QueryPointer:
      case 15:  // X_QueryTree:
	{
	  messageLength=8;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  dBuffer.decodeCachedValue(wvalue, 29,
				    windowCache, 9);
	  wBuffer.putULONG(wvalue[0], 4);
	  if((iopcode==38/*X_QueryPointer*/) ||
	     (iopcode==3/*X_GetWindowAttributes*/) ||
	     (iopcode==15/*X_QueryTree*/)){
	    sNumQueue.push(cCache.lastRequestSequenceNum, iopcode);
	  }
	}
	break;
      case 45: // X_OpenFont
	{
	  dBuffer.decodeValue(wvalue, 16, 7);
	  int nameLength=wvalue[0];
	  messageLength = (15+nameLength)/4*4;
	  messageStart = wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
          wBuffer.putUINT(nameLength, 8);
	  dBuffer.decodeValue(wvalue, 29, 5);
	  cCache.lastFont += wvalue[0];
	  cCache.lastFont &= 0x1fffffff;
          wBuffer.putULONG(cCache.lastFont, 4);
	  int nextDest = messageStart+12;
	  cCache.openFontTextCompressor.reset();
	  for (; nameLength!=0; nameLength--){
	    message[nextDest++] =
	      cCache.openFontTextCompressor.decodeChar(dBuffer);
	  }
	}
	break;
      case 70: // X_PolyFillRectangle
	{
	  messageLength = 12;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  wBuffer.registerPointer(messageStart);
	  dBuffer.decodeCachedValue(wvalue, 29,
				    drawableCache, 9);
	  wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 29, gcCache, 9);
	  wBuffer.putULONG(wvalue[0], 8);
	  int index=0;
	  int lastX=0, lastY=0, lastWidth=0, lastHeight=0;
	  int numRectangles = 0;
	  while(true){
	    messageLength+=8;
	    wBuffer.addMessage(8);
	    message=wBuffer.getData();
	    int nextDest = 0;
	    numRectangles++;
	    dBuffer.decodeCachedValue(wvalue, 16,
				      cCache.polyFillRectangleCacheX[index], 8);
	    wvalue[0]+=lastX;
	    wBuffer.putUINT(wvalue[0], nextDest);
	    lastX = wvalue[0];
	    nextDest += 2;
	    dBuffer.decodeCachedValue(wvalue, 16,
				      cCache.polyFillRectangleCacheY[index], 8);
	    wvalue[0]+=lastY;
	    wBuffer.putUINT(wvalue[0], nextDest);
	    lastY = wvalue[0];
	    nextDest += 2;
	    dBuffer.decodeCachedValue(wvalue, 16,
				      cCache.polyFillRectangleCacheWidth[index], 8);
	    wvalue[0] += lastWidth;
	    wBuffer.putUINT(wvalue[0], nextDest);
	    lastWidth = wvalue[0];
	    nextDest += 2;
	    dBuffer.decodeCachedValue(wvalue, 16,
				      cCache.polyFillRectangleCacheHeight[index], 8);
	    wvalue[0] += lastHeight;
	    wBuffer.putUINT(wvalue[0], nextDest);
	    lastHeight = wvalue[0];
	    nextDest += 2;
	    index = 1;
	    dBuffer.decodeValue(wvalue, 1);
	    if(wvalue[0]==0)
	      break;
	  }
	  wBuffer.unregisterPointer();
	}
	break;
      case 64: // X_PolyPoint:
	{
	  dBuffer.decodeValue(wvalue, 16, 4);
	  int numPoints=wvalue[0];
	  messageLength=(numPoints<<2)+12;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  dBuffer.decodeValue(wvalue, 1);
	  int relativeCoordMode=wvalue[0];
	  message[messageStart+1]=(byte)relativeCoordMode;
	  dBuffer.decodeCachedValue(wvalue, 29, drawableCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 29, gcCache, 9);
          wBuffer.putULONG(wvalue[0], 8);
	  int nextDest=/*outputMessage+*/12;
	  int index=0;
	  int lastX=0, lastY=0;
	  for(int i=0; i<numPoints; i++){
	    dBuffer.decodeCachedValue(wvalue, 16,
				      cCache.polyPointCacheX[index], 8);
	    lastX+=wvalue[0];
	    wBuffer.putUINT(lastX, nextDest);
	    nextDest+=2;
	    dBuffer.decodeCachedValue(wvalue, 16,
				      cCache.polyPointCacheY[index], 8);
	    lastY+=wvalue[0];
	    wBuffer.putUINT(lastY, nextDest);
	    nextDest+=2;
	    index=1;
	  }
	}
	break;
      case 65: // X_PolyLine:
	{
	  dBuffer.decodeValue(wvalue, 16, 4);
	  int numPoints=wvalue[0];
	  messageLength=(numPoints<<2)+12;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  dBuffer.decodeValue(wvalue, 1);
	  int relativeCoordMode=wvalue[0];
	  message[messageStart+1]=(byte)relativeCoordMode;
	  dBuffer.decodeCachedValue(wvalue, 29, drawableCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 29, gcCache, 9);
          wBuffer.putULONG(wvalue[0], 8);
	  int nextDest=/*messageStart+*/12;
	  int index = 0;
	  int lastX = 0, lastY = 0;
	  for(int i=0; i<numPoints; i++)
	  {
	    dBuffer.decodeCachedValue(wvalue, 16,
				      cCache.polyLineCacheX[index], 8);
	    lastX+=wvalue[0];
            wBuffer.putUINT(lastX, nextDest);
	    nextDest+=2;
	    dBuffer.decodeCachedValue(wvalue, 16,
				      cCache.polyLineCacheY[index], 8);
	    lastY+=wvalue[0];
            wBuffer.putUINT(lastY, nextDest);
	    nextDest+=2;
	    index=1;
	  }
	}
	break;
      case 67: // X_PolyRectangle:
	{
	  dBuffer.decodeValue(wvalue, 16, 3);
          int numRectangles=wvalue[0];
	  messageLength=(numRectangles<<3)+12;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeCachedValue(wvalue, 29, drawableCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 29, gcCache, 9);
          wBuffer.putULONG(wvalue[0], 8);
	  int nextDest=/*outputMessage+*/12;
	  for(int i=0; i<numRectangles; i++){
	    for(int k=0; k<4; k++){
	      dBuffer.decodeCachedValue(wvalue, 16,
					cCache.polyRectangleGeomCache[k], 8);
	      wBuffer.putUINT(wvalue[0], nextDest);
	      nextDest+=2;
	    }
	  }
	}
	break;
      case 66: // X_PolySegment:
	{
	  dBuffer.decodeValue(wvalue, 16, 4);
	  int numSegments=wvalue[0];
	  messageLength=(numSegments<<3)+12;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  dBuffer.decodeCachedValue(wvalue, 29, drawableCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 29, gcCache, 9);
          wBuffer.putULONG(wvalue[0], 8);
	  int nextDest=/*outputMessage+*/ 12;
	  for(numSegments*=2;numSegments>0; numSegments--){
	    dBuffer.decodeValue(wvalue, 1);
	    int index=wvalue[0];
	    dBuffer.decodeCachedValue(wvalue, 16, cCache.polySegmentCacheX, 6);
	    int x=wvalue[0];
	    x+=cCache.polySegmentLastX[index];
            wBuffer.putUINT(x, nextDest);
	    nextDest+=2;

	    dBuffer.decodeCachedValue(wvalue, 16, cCache.polySegmentCacheY, 6);
	    int y=wvalue[0];
	    y+=cCache.polySegmentLastY[index];
            wBuffer.putUINT(y, nextDest);
	    nextDest+=2;

	    cCache.polySegmentLastX[cCache.polySegmentCacheIndex]=x;
	    cCache.polySegmentLastY[cCache.polySegmentCacheIndex]=y;
	    if(cCache.polySegmentCacheIndex==1){
	      cCache.polySegmentCacheIndex=0;
	    }
	    else{
	      cCache.polySegmentCacheIndex=1;
	    }
	  }
	}
	break;
      case 74: // X_PolyText8:
	{
	  messageLength = 16;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  dBuffer.decodeCachedValue(wvalue, 29, drawableCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 29, gcCache, 9);
          wBuffer.putULONG(wvalue[0], 8);
	  dBuffer.decodeCachedValue(wvalue, 16, cCache.polyText8CacheX, 8);
	  cCache.polyText8LastX+=wvalue[0];
	  cCache.polyText8LastX&=0xffff;
          wBuffer.putUINT(cCache.polyText8LastX, 12);
	  dBuffer.decodeCachedValue(wvalue, 16, cCache.polyText8CacheY, 8);
	  cCache.polyText8LastY+=wvalue[0];
	  cCache.polyText8LastY&=0xffff;
          wBuffer.putUINT(cCache.polyText8LastY, 14);
	  int addedLength = 0;
	  wBuffer.registerPointer(messageStart);
	  while(true){
	    dBuffer.decodeValue(wvalue, 1);
	    if(wvalue[0]==0) break;
	    dBuffer.decodeValue(wvalue, 8);
	    int textLength=wvalue[0];
	    if(textLength==255){
	      addedLength+=5;
	      int nextSegment=wBuffer.addMessage(5);
	      message=wBuffer.getData();
	      message[nextSegment]=(byte)textLength;
	      dBuffer.decodeCachedValue(wvalue, 29, cCache.polyText8FontCache);
	      wBuffer.putULONG(wvalue[0], 1);
	    }
	    else {
	      addedLength+=(textLength+2);
	      int nextSegment=wBuffer.addMessage(textLength + 2);
	      message=wBuffer.getData();
	      message[nextSegment]=(byte)textLength;
	      int nextDest=nextSegment+1;
	      dBuffer.decodeCachedValue(cValue, 8, cCache.polyText8DeltaCache);
	      message[nextDest++]=cValue[0];
	      cCache.polyText8TextCompressor.reset();
	      while(textLength!=0){
		message[nextDest++]=
		  cCache.polyText8TextCompressor.decodeChar(dBuffer);
		textLength--;
	      }
	    }
	  }
	  messageLength+=addedLength;
	  int mod4=(addedLength&0x3);
	  if(mod4!=0){
	    int extra=4-mod4;
	    int nextDest=wBuffer.addMessage(extra);
	    message=wBuffer.getData();
	    for(int i=0; i<extra; i++)
	      message[nextDest++]=0;
	    messageLength+=extra;
	  }
	  wBuffer.unregisterPointer();
	}
	break;
      case 72: // X_PutImage
	{
	  dBuffer.decodeValue(wvalue, 16, 8);
	  messageLength=(wvalue[0] << 2);
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  dBuffer.decodeValue(wvalue, 2);
	  message[messageStart+1] = (byte)wvalue[0];
	  dBuffer.decodeCachedValue(wvalue, 29,
				    drawableCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 29, gcCache, 9);
          wBuffer.putULONG(wvalue[0], 8);
	  dBuffer.decodeCachedValue(wvalue, 16,
				    cCache.putImageWidthCache, 8);
	  int width=wvalue[0];
          wBuffer.putUINT(width, 12);

	  dBuffer.decodeCachedValue(wvalue, 16, cCache.putImageHeightCache, 8);
	  int height=wvalue[0];
          wBuffer.putUINT(height, 14);
	  dBuffer.decodeCachedValue(wvalue, 16,
				    cCache.putImageXCache, 8);
	  cCache.putImageLastX+=wvalue[0];
	  cCache.putImageLastX&=0xffff;
          wBuffer.putUINT(cCache.putImageLastX, 16);
	  dBuffer.decodeCachedValue(wvalue, 16, cCache.putImageYCache, 8);
	  cCache.putImageLastY+=wvalue[0];
	  cCache.putImageLastY&=0xffff;
          wBuffer.putUINT(cCache.putImageLastY, 18);
	  dBuffer.decodeCachedValue(cValue, 8,
				    cCache.putImageOffsetCache);
	  message[messageStart+20] = cValue[0];
	  dBuffer.decodeCachedValue(cValue, 8, cCache.depthCache);
	  message[messageStart+21] = cValue[0];
	  int nextDest = messageStart+24;
//System.out.println("1#");
	  if ((message[messageStart+1] == 0) && (height <= 32) &&
	      (width > (height * Constants.PUT_IMAGE_MIN_ASPECT_RATIO))){
//System.out.println("2#");
	    // bitmap that probably contains text; encode using a variant of
	    // text-compression algorithm
            int widthInBits=((width/scanlinePad)*scanlinePad);
            if(widthInBits<width)
	      widthInBits+=scanlinePad;
            int widthInBytes=(widthInBits>>>3);
	    int foo = messageStart + 24;
	    for(int i=0; i<messageLength-24;i++) message[i]=0;
	    int destMask = 0x80;
	    cCache.putImageLastPixels.reset();
            for(int xCoord = 0; xCoord < width; xCoord++){
	      int modelNum =
		cCache.putImageLastPixels.getValue();
	      dBuffer.decodeCachedValue(wvalue, height,
					cCache.putImagePixelCache[modelNum %
								  ClientCache.PUT_IMAGE_PIXEL_CACHE_SIZE],
					cCache.columnPixel0Coder,
					cCache.columnPixel1Coder);
	      int columnValue=wvalue[0];
	      int next = foo;
	      int mask = (1 << (height - 1));
	      for (int h = 0; h < height; h++){
		if ((columnValue & mask)!=0)
		  message[next] |= destMask;
		next += widthInBytes;
		mask >>>= 1;
	      }
	      destMask >>>= 1;
	      if (destMask == 0){
		destMask = 0x80;
		nextDest++;
	      }
	      cCache.putImageLastPixels.add(columnValue);
	    }
	    if ((imageByteOrder == 0) && (bitmapBitOrder == 0)){
	      int next = messageStart+24;
	      for(int i = 24; i < messageLength; i++){
		message[next]=Constants.REVERSED_BYTE[message[next]];
		next++;
	      }
	    }
	  }
	  else if(message[messageStart+1]==0){
//System.out.println("3#");
            // bitmap--use "Modified-Modified-Read" FAX coding
	    if (width+2>cCache.putImageLineSize){
//System.out.println("33#");
//	      delete[]clientCache_.putImageReferenceLine;
//	      delete[]clientCache_.putImageCodingLine;
	      cCache.putImageLineSize=width+2;
	      cCache.putImageReferenceLine=new int[width+2];
	      cCache.putImageCodingLine=new /*unsigned*/ int[width + 2];
	    }
	    int widthInBits = ((width / scanlinePad) * scanlinePad);
	    if (widthInBits < width)
	      widthInBits += scanlinePad;
//System.out.println("widthInBits="+widthInBits);
	    int widthInBytes = (widthInBits >>> 3);
//System.out.println("widthInBytes="+widthInBytes);
	    int lastPixelValue = 0;
	    for (int h = 0; h < height; h++){
//System.out.println("h="+h);
	      int codingLineLength = 0;
	      int nextDest2 = messageStart + 24 + h * widthInBytes;
	      message[/*messageStart+*/nextDest2] = 0;
	      int destMask = 0x80;
	      if (h == 0){
		dBuffer.decodeValue(wvalue, 1);
		int pixelValue=wvalue[0];
//System.out.println("pixelValue="+pixelValue);
		for (int xCoord = 0; xCoord < width;){
		  if (pixelValue!=0){
		    if (pixelValue != lastPixelValue){
		      cCache.putImageCodingLine[codingLineLength++] =xCoord;
		    }
		    int runLength = cCache.putImagePixel1Coder.decode(dBuffer);
//System.out.println("1 runLength="+runLength);
		    while (runLength--!=0){
		      message[nextDest2] |= destMask;
		      destMask=((destMask>>>1)&0x7f);
		      if (destMask == 0){
			destMask = 0x80;
			nextDest2++;
			if (xCoord + 1 < width)
			  message[nextDest2] = 0;
		      }
		      xCoord++;
		    }
		    pixelValue = 0;
		    lastPixelValue = 1;
		  }
		  else{
		    if (pixelValue != lastPixelValue){
		      cCache.putImageCodingLine[codingLineLength++] =xCoord;
		    }
		    int runLength=cCache.putImagePixel0Coder.decode(dBuffer);
//System.out.println("2 runLength="+runLength);
		    while (runLength-->0){
		      destMask=((destMask>>>1)&0x7f);
		      if (destMask == 0){
			destMask = 0x80;
			nextDest2++;
			if (xCoord + 1 < width)
			  message[nextDest2] = 0;
		      }
		      xCoord++;
		    }
		    pixelValue = 1;
		    lastPixelValue = 0;
		  }
		}
		cCache.putImageCodingLine[codingLineLength++] = width;
	      }
	      else{
		int lastX = 0;
		int nextReferenceIndex = 0;
		while (lastX < width) {
                  int diffCode=cCache.putImageDiffCoder.decode(dBuffer);
//System.out.println("diffCode="+diffCode+", lastX="+lastX);
                  switch (diffCode){
		  case ClientCache.SD_VERTICAL_0:
		    {
		      lastX =
			cCache.putImageCodingLine[codingLineLength++] =
			cCache.putImageReferenceLine[nextReferenceIndex++];
		    }
		    break;
		  case ClientCache.SD_VERTICAL_PLUS_1:
		    {
		      lastX =
			cCache.putImageCodingLine[codingLineLength++] =
			cCache.putImageReferenceLine[nextReferenceIndex++] + 1;
		    }
		    break;
		  case ClientCache.SD_VERTICAL_PLUS_2:
		    {
		      lastX =
			cCache.putImageCodingLine[codingLineLength++] =
			cCache.putImageReferenceLine[nextReferenceIndex++] + 2;
		    }
		    break;
		  case ClientCache.SD_VERTICAL_MINUS_1:
		    {
		      lastX =
			cCache.putImageCodingLine[codingLineLength++] =
			cCache.putImageReferenceLine[nextReferenceIndex++] - 1;
		    }
		    break;
		  case ClientCache.SD_VERTICAL_MINUS_2:
		    {
		      lastX =
			cCache.putImageCodingLine[codingLineLength++] =
			cCache.putImageReferenceLine[nextReferenceIndex++] - 2;
		    }
		    break;
		  case ClientCache.SD_PASS:
		    {
		      nextReferenceIndex += 2;
		    }
		    break;
		  case ClientCache.SD_HORIZONTAL:
		    {
		      int diff;
		      if ((codingLineLength & 1)!=0)
			diff = cCache.putImagePixel0Coder.decode(dBuffer);
		      else
			diff = cCache.putImagePixel1Coder.decode(dBuffer);
		      lastX += diff;
		      lastX &= 0xffff;
		      cCache.putImageCodingLine[codingLineLength++] = lastX;
		      if ((codingLineLength & 1)!=0)
			diff = cCache.putImagePixel0Coder.decode(dBuffer);
		      else
			diff = cCache.putImagePixel1Coder.decode(dBuffer);
		      lastX += diff;
		      lastX &= 0xffff;
		      cCache.putImageCodingLine[codingLineLength++]=lastX;
		    }
		  default:
		    {
		    }
		  }
                }
	      }

	      cCache.putImageCodingLine[codingLineLength++] = width;
	      int pixelValue = 0;
	      int lastPixelChange = 0;
	      int nextPixelChange = 0;
	      for (int xCoord = 0; xCoord < width;){
		int count = cCache.putImageCodingLine[nextPixelChange] - lastPixelChange;
		lastPixelChange = cCache.putImageCodingLine[nextPixelChange];
		nextPixelChange++;

		for (; count!=0; count--){
		  if (pixelValue!=0)
		    message[nextDest2] |= destMask;
		  destMask=((destMask>>>1)&0x7f);
		  if (destMask == 0) {
		    destMask = 0x80;
		    nextDest2++;
		    if (xCoord + 1 < width)
		      message[nextDest2] = 0;
		  }
		  xCoord++;
		}
		if (pixelValue!=0) pixelValue = 0;
		else pixelValue = 1;
              }

	      int[] tmp = cCache.putImageReferenceLine;
	      cCache.putImageReferenceLine =cCache.putImageCodingLine;
	      cCache.putImageCodingLine = tmp;
	    }

	    int end = messageStart + messageLength;
	    if ((imageByteOrder == 0) && (bitmapBitOrder == 0)) {
	      for (int next = messageStart+24; next < end;  next++){
		message[next]=Constants.REVERSED_BYTE[(message[next])&0xff];
	      }
	    }
	    int next = messageStart+24+ widthInBytes;
	    int prev = messageStart+24;
	    for (; next < end;) message[next++] ^= message[prev++];
	  }
	  else {
//System.out.println("4#");
	    // pixmap
	    if(message[messageStart+21] == 8) {
	      for(int i = 24; i < messageLength; i++){
		dBuffer.decodeCachedValue(cValue, 8,
					  cCache.putImageByteCache, 4);
		message[nextDest++] = cValue[0];
	      }
	    }
	    else{
	      for(int i = 24; i < messageLength; i++){
		dBuffer.decodeValue(wvalue, 8);
		message[nextDest++]=(byte)wvalue[0];
	      }
	    }
	  }
	}
	break;
      case 97: // QueryBestSize
	{
	  messageLength=12;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  dBuffer.decodeValue(wvalue, 2);
	  message[messageStart+1]=(byte)wvalue[0];
	  dBuffer.decodeCachedValue(wvalue, 29,
				    drawableCache, 9);
	  wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeValue(wvalue, 16, 8);
	  wBuffer.putUINT(wvalue[0], 8);
	  dBuffer.decodeValue(wvalue, 16, 8);
	  wBuffer.putUINT(wvalue[0], 10);
	  sNumQueue.push(cCache.lastRequestSequenceNum, iopcode);
	}
	break;
      case 91: // X_QueryColors:
	{
	  dBuffer.decodeValue(wvalue, 16, 5);
	  int numColors=wvalue[0];
	  messageLength=(numColors<<2)+8;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  dBuffer.decodeCachedValue(wvalue, 29, cCache.colormapCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  int nextDest = /*messageStart+*/8;
	  int predictedPixel = cCache.queryColorsLastPixel;
	  for(int i=0; i<numColors; i++){
	    int pixel;
	    dBuffer.decodeValue(wvalue, 1);
	    if(wvalue[0]!=0){
	      pixel = predictedPixel;
	    }
	    else{
	      dBuffer.decodeValue(wvalue, 32, 9);
	      pixel=wvalue[0];
	    }
            wBuffer.putULONG(pixel, nextDest);
	    if(i==0){
	      cCache.queryColorsLastPixel = pixel;
	    }
	    predictedPixel=pixel+1;
	    nextDest+=4;
	  }
	  sNumQueue.push(cCache.lastRequestSequenceNum, iopcode);
	}
	break;
      case 98: // QueryExtension
	{
	  dBuffer.decodeValue(wvalue, 16, 6);
	  int nameLength=wvalue[0];
	    //System.out.println("nameLength="+nameLength);
	  messageLength=8+(nameLength+3)/4*4;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  wBuffer.putUINT(nameLength, 4);
	  int nextDest=messageStart+8;
	  for (int i=0; i<nameLength; i++){
	    dBuffer.decodeValue(wvalue, 8);
	    message[nextDest++]=(byte) wvalue[0];
	  }
	  int hideExtension=0;
	  byte[] mitshm="MIT-SHM".getBytes();
	  if(messageLength-(messageStart+8)>=7){
	    hideExtension=1;
	    for(int i=0; i<7; i++){
              if(message[messageStart+8+i]==mitshm[i]) continue;
	      hideExtension=0;
	      break;
	    }
	  }
	  sNumQueue.push(cCache.lastRequestSequenceNum, iopcode,
			 hideExtension);
	}
	break;
      case 47: // X_QueryFont:
	{
	  messageLength = 8;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  dBuffer.decodeValue(wvalue, 29, 5);
	  cCache.lastFont+=wvalue[0];
	  cCache.lastFont&=0x1fffffff;
          wBuffer.putULONG(cCache.lastFont, 4);
	  sNumQueue.push(cCache.lastRequestSequenceNum, iopcode);
	}
	break;
      case 59: // X_SetClipRectangles:
	{
	  dBuffer.decodeValue(wvalue, 13, 4);
	  int numRectangles=wvalue[0];
	  messageLength=(numRectangles<<3)+12;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeValue(wvalue, 2);
	  message[messageStart+1]=(byte)wvalue[0];
	  dBuffer.decodeCachedValue(wvalue, 29, gcCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 16,
				    cCache.setClipRectanglesXCache, 8);
          wBuffer.putUINT(wvalue[0], 8);
	  dBuffer.decodeCachedValue(wvalue, 16,
				    cCache.setClipRectanglesYCache, 8);
          wBuffer.putUINT(wvalue[0], 10);
	  int nextDest=/*outputMessage+*/12;
	  for(int i=0; i<numRectangles; i++){
	    for(int k=0; k<4; k++){
	      dBuffer.decodeCachedValue(wvalue, 16,
					cCache.setClipRectanglesGeomCache[k], 8);
	      wBuffer.putUINT(wvalue[0], nextDest);
	      nextDest+=2;
	    }
	  }
	}
	break;
      case 58: // X_SetDashes:
	{
	  dBuffer.decodeCachedValue(wvalue, 16,
				    cCache.setDashesLengthCache, 5);
	  int numDashes=wvalue[0];
	  messageLength=12+(numDashes+3)/4*4;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  wBuffer.putUINT(numDashes, 10);
	  dBuffer.decodeCachedValue(wvalue, 29, gcCache, 9);
	  wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 16,
				    cCache.setDashesOffsetCache, 5);
	  wBuffer.putUINT(wvalue[0], 8);
	  int nextDest=messageStart+12;
	  for(int i=0; i<numDashes; i++){
	    dBuffer.decodeCachedValue(cValue, 8,
				      cCache.setDashesDashCache[i&1], 5);
	    message[nextDest++]=cValue[0];
	  }
	}
	break;
      case 22: // X_SetSelectionOwner:
	{
	  messageLength=16;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeCachedValue(wvalue, 29, cCache.setSelectionOwnerCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 29,
				    cCache.getSelectionOwnerSelectionCache, 9);
          wBuffer.putULONG(wvalue[0], 8);
	  dBuffer.decodeCachedValue(wvalue, 32,
				    cCache.setSelectionOwnerTimestampCache, 9);
          wBuffer.putULONG(wvalue[0], 12);
	}
	break;
      case 40: //X_TranslateCoords:
	{
	  messageLength=16;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  dBuffer.decodeCachedValue(wvalue, 29,
				    cCache.translateCoordsSrcCache, 9);
          wBuffer.putULONG(wvalue[0], 4);
	  dBuffer.decodeCachedValue(wvalue, 29,
				    cCache.translateCoordsDestCache, 9);
          wBuffer.putULONG(wvalue[0], 8);
	  dBuffer.decodeCachedValue(wvalue, 16,
				    cCache.translateCoordsXCache, 8);
          wBuffer.putULONG(wvalue[0], 12);
	  dBuffer.decodeCachedValue(wvalue, 16,
				    cCache.translateCoordsYCache, 8);
          wBuffer.putULONG(wvalue[0], 14);
	  sNumQueue.push(cCache.lastRequestSequenceNum, iopcode);
	}
	break;
      default:
	{
/*
	  if((opcode[0]&0xff)<=127 
	     && opcode[0]!= 7
	     && opcode[0]!=25
	     && opcode[0]!=27
	     && opcode[0]!=30
 	     && opcode[0]!=32
 	     && opcode[0]!=33
 	     && opcode[0]!=35
	     && opcode[0]!=42
	     && opcode[0]!=68
	     && opcode[0]!=71
	     && opcode[0]!=75
	     && opcode[0]!=73
	     && opcode[0]!=81
	     && opcode[0]!=86
	     && opcode[0]!=93
	     && opcode[0]!=94
	     && opcode[0]!=96
	     && opcode[0]!=102
	     && opcode[0]!=103
	     && opcode[0]!=104
	     && opcode[0]!=112
	     ){
	    System.out.println("unsupported request: "+(opcode[0]&0xff));
	  }
*/
	  dBuffer.decodeValue(wvalue, 8);
	  int secondByte=wvalue[0];
	  dBuffer.decodeValue(wvalue, 16, 8);
	  messageLength=wvalue[0];
	  messageLength<<=2;
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();
	  message[messageStart+1]=(byte)secondByte;
	  int nextDest=messageStart+4;
	  for (int i=4; i<messageLength; i++) {
	    dBuffer.decodeValue(wvalue, 8);
	    message[nextDest++]=(byte)wvalue[0];
	  }
	}
      }
      message[messageStart]=opcode[0];
      wBuffer.putUINT(messageLength>>>2, 2);
      putByte(message, messageStart, messageLength);
    }
  }

  void setBigEndian(boolean foo){
    super.setBigEndian(foo);
    rBuffer.setBigEndian(foo);
    wBuffer.setBigEndian(foo);
  }
}
