/* JDxpc -- DXPC in pure Java
 * 
 *  MultiClientProxy
 *
 *  Copyright (C) 2000 Marcus Schiesser <info@linuxtest.de> 
 *  This allows to run multiple client proxies on a server in one process
 *  To spawn new clientproxies you have to connect to a server socket
 *  the first command line is the port of this server socket
 *  How to spawn a new client:
 *  1. connect to multiclientproxy
 *  2. send password
 *  3. if password is ok 'OK' is sent
 *  4. send command (could be anything, there is only one command so far - for future compatibility use "SPAWN"
 *  5. send host the clientproxy should connect to
 *  6. send encryption session key (not used yet)
 *  7. if there is a free x-port, 'OK' is returned and the number of the free x-port
 *
 *  The whole thing takes care of the occupied xports
 *  There is a timeout value to allow the clients to start
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package com.jcraft.jdxpc;

import java.io.*;
import java.net.*;
import java.lang.*;

class MultiClientProxy extends Thread{

  static class Port {
    Port(int nr, boolean free){
      this.nr = nr;
      this.free = free;
      next = null;
      previous = null;
    }
    int nr;
    boolean free;
    Port next, previous;
  }

  static Port head = null, tail = null;
  final static String passwd = "apemaster";
  final static int startport=6001, endport=6099;
  final static int jdxpcport=4003;
  final static int timeout=30000;

  private Socket socket;

  MultiClientProxy(Socket socket){
    this.socket = socket;
    start();
  }

  synchronized private void freeport(Port port){
    // Free the port (move it to front)
    port.free = true;
    // delink
    if(port.previous != null){
      port.previous.next = port.next;
    }
    if(port.next != null){
      port.next.previous = port.previous;
    }
    // move to front
    port.previous = null;
    port.next = head;
    head = port;
  }

  // returns 0 if no xport is available
  synchronized private Port getfreeport(){
    // check whether there is a free port
    if(!head.free) return null;
    // occupy port by moving it to tail
    Port foo = head;
    foo.free = false;

    head = head.next;
    head.previous = null;

    tail.next = foo;
    foo.previous = tail;
    foo.next = null;
    tail = foo;

    return foo;
  }

  public void run(){
    BufferedReader in = null;
    PrintStream out = null;
    try{
      in=new BufferedReader( new InputStreamReader( socket.getInputStream()));
      out=new PrintStream( socket.getOutputStream() );
    }
    catch(IOException e){
      try {
        socket.close();
      }
      catch (IOException ie){ }
      return;
    }
    Port xport=null;
    String host=null, code, reserved;
    ClientProxy cp=null;
    try {
      String pass = in.readLine();
      if(!pass.equals(passwd)){
        out.println("BADPW");
      }
      else{
        out.println("OK");
        // reserved for a commando string 
        // (if more then one commando should be supported
	reserved = in.readLine();
	// get host
	host = in.readLine();
	// get encryption code (for future encryption)
	code = in.readLine();
	// look for free x-port
	xport = getfreeport();
	if(xport == null){
	  out.println("NOFREEPORT");
	}
	else {
	  // wait till jdxpc server is available on client side 
          // (in fact its a server)
          try{
            sleep(timeout);
	  }
	  catch(InterruptedException e){ }
	  boolean ok=true;
	  try{
            cp = new ClientProxy(host, xport.nr, jdxpcport);
	  }
	  catch( IOException e){
            out.println("NOJDXPC");
	    ok=false;
	    cp=null;
	  }
	  if(ok){
            out.println("OK");
	    // send port number
	    String sport = Integer.toString( xport.nr );
	    out.println(sport);
	  }
	}
      }
    }
    catch (IOException ie) {
    }
    finally{
      try{
        in.close();
	out.close();
	socket.close();
      }
      catch (IOException ie) { }
    }
    if(xport!=null){
      if(cp!=null){
	cp.run();
      }
      freeport(xport);
    }
  }

  public static void main(String[] arg){
    // usage: port 
    if(arg.length!=1) return;

    // create occupy data structure
    Port act,previous;
    head = act = previous = new Port(startport, true);
    for(int i=startport+1; i<=endport; i++){
	act = new Port(i, true);
	previous.next = act;
	act.previous = previous;
	previous = act;
    }
    tail = act;

    // start server
    ServerSocket server;
    try{
      server = new ServerSocket(Integer.parseInt(arg[0]));
    }
    catch(IOException e) {
      System.err.println("Unable to open socket");
      e.printStackTrace();
      return;
    }
    try {
      while(true) {
        new MultiClientProxy(server.accept());
      }
    }
    catch (IOException e) { }
  }
}
