/* JDxpc -- DXPC in pure Java
 *
 *  Copyright (C) 2000 ymnk, JCraft, Inc.
 *
 *  Many thanks to 
 *    Brian Pane<brianp@cnet.com> and
 *    Zachary Vonler<lightborn@mail.utexas.edu>.
 *  JDxpc has been based on their awesome works, dxpc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package com.jcraft.jdxpc;

class HuffmanCoder{
  private EncodeNode root;
  int numTokens;
  EncodeNode[] tokens;
  HuffmanCoder(int histogramLength, int[] histogram){
    this(histogramLength, histogram, 0);
  }
  HuffmanCoder(int histogramLength, int[] histogram, int overflowCount){
    numTokens=histogramLength;
    tokens=new EncodeNode[histogramLength+1];
    Heap heap=new Heap(histogramLength+1);
    for(int i=0; i<histogramLength; i++){
      heap.insert(tokens[i]=new EncodeNode(i,histogram[i]));
    }
    heap.insert(tokens[histogramLength] =
		new EncodeNode(histogramLength, overflowCount));
    while(heap.getSize()>1){
      EncodeNode node1 = heap.pop();
      EncodeNode node2 = heap.pop();
      EncodeNode newNode = new EncodeNode(node1, node2, 
					  node1.getFrequency()+node2.getFrequency());
      heap.insert(newNode);
    }
    root=heap.pop();
    root.setCode(null, 0);
  }
  HuffmanCoder(int numCodes, byte[][] codes){
    numTokens=numCodes;
    tokens=new EncodeNode[numCodes];
    root=new EncodeNode(null, null, 0);
    for(int i=0; i<numCodes; i++){
      tokens[i] = root.addCode(i, codes[i], 0);
    }
    root.setCode(null, 0);
  }
  void encode(int value, EncodeBuffer encodeBuffer){
    int index=value;
    if(index>=numTokens) index=numTokens;
    EncodeNode node = tokens[index];
    byte[] code = node.getCode();
    int codeLength=node.getCodeLength();
    for(int i=0; i<codeLength; i++)
      encodeBuffer.encodeValue(code[i], 1);
    if(value>=numTokens)
      encodeBuffer.encodeValue(value, 16, 8);
  }
  int decode(DecodeBuffer decodeBuffer){
    int[] result=new int[1];
    result[0]=root.decode(decodeBuffer);
    if (result[0]>=numTokens) decodeBuffer.decodeValue(result, 16, 8);
    return result[0];
  }

  class EncodeNode{
    private int value;
    private int frequency;
    private byte[] encoding=null;
    private int codeLength=0;	// in bits

    private EncodeNode left;
    private EncodeNode right;

    EncodeNode(int v, int f){
      value=v;
      frequency=f;
      left=null;
      right=null;
    }
    EncodeNode(EncodeNode left, EncodeNode right, int f){
      value=0;
      frequency=f;
      this.left=left;
      this.right=right;
    }
    int getValue(){ return value; }
    int getFrequency() { return frequency; }
    void setCode(byte[] code, int length){
      encoding = code;
      codeLength = length;
      int nextCodeLength = codeLength + 1;
      if(left!=null){
	byte[] leftCode = new byte[nextCodeLength];
	int i=0;
	for(; i<codeLength; i++) leftCode[i]=encoding[i];
	leftCode[i]=0;
	left.setCode(leftCode, nextCodeLength);
      }
      if(right!=null){
	byte[] rightCode = new byte[nextCodeLength];
	int i = 0;
	for(;i<codeLength; i++) rightCode[i]=encoding[i];
	rightCode[i]=1;
	right.setCode(rightCode, nextCodeLength);
      }
    }

    int getCodeLength(){ return codeLength; }

    byte[] getCode(){ return encoding; }

    EncodeNode addCode(int v, byte[] code, int start){
      if(code.length==start){
	value=v;
	return this;
      }
      else if (code[start]=='0'){
	if(left==null) left=new EncodeNode(null, null, 0);
	return left.addCode(v, code, start+1);
      }
      else{
	if(right==null) right=new EncodeNode(null, null, 0);
	return right.addCode(v, code, start+1);
      }
    }

    int[] nextBit=new int[1];
    int decode(DecodeBuffer decodeBuffer){
      if ((left==null)&&(right==null)) return value;
      //int[] nextBit=new int[1];
      decodeBuffer.decodeValue(nextBit, 1);
      if(nextBit[0]==0) return left.decode(decodeBuffer);
      else  return right.decode(decodeBuffer);
    }
  }

  class Heap{
    private int numElements=0;
    private EncodeNode[] heap;

    Heap(int maxSize){
      heap=new EncodeNode[maxSize];
    }
    void insert(EncodeNode node){
      int location = numElements++;
      int parent=((location-1)>>1);
      while(location>0 && (heap[parent].getFrequency()>node.getFrequency())){
	heap[location]=heap[parent];
	location=parent;
	parent=((location-1)>>1);
      }
      heap[location]=node;
    }
    EncodeNode pop(){
      if(numElements==0) return null;
      EncodeNode result=heap[0];
      numElements--;
      EncodeNode node=heap[numElements];
      int location = 0;
      do{
	int left=(location<<1)+1;
	if(left>=numElements) break;
	int nodeToSwap=left;
	int right=left+1;
	if(right<numElements){
	  if(heap[right].getFrequency()<heap[left].getFrequency()){
	    nodeToSwap=right;
	  }
	}
	if(heap[nodeToSwap].getFrequency()<node.getFrequency()){
	  heap[location]=heap[nodeToSwap];
	  location=nodeToSwap;
	}
	else{
	  break;
	}
      }
      while(location<numElements);
      heap[location]=node;
      return result;
    }
    int getSize(){ return numElements; }
  }
}
