/* JDxpc -- DXPC in pure Java
 *
 *  Copyright (C) 2000 ymnk, JCraft, Inc.
 *
 *  Many thanks to 
 *    Brian Pane<brianp@cnet.com> and
 *    Zachary Vonler<lightborn@mail.utexas.edu>.
 *  JDxpc has been based on their awesome works, dxpc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package com.jcraft.jdxpc;

class EncodeBuffer{
  static final int INITIAL_BUFFER_SIZE=256;
  static final int PREFIX_SIZE=16;

  int size=INITIAL_BUFFER_SIZE;
  int nextDest;
  int destMask;
  int cumulativeBits;
  byte[] buffer=new byte[INITIAL_BUFFER_SIZE+PREFIX_SIZE];

  EncodeBuffer(){
    reset();
  }

  void reset(){
    nextDest=PREFIX_SIZE;
    destMask=0x80;
    cumulativeBits=0;
    buffer[nextDest]=(byte)0;
  }
  void encodeValue(int value, int numBits){ encodeValue(value, numBits, 0); }
  void encodeValue(int value, int numBits, int blockSize){
    int srcMask = 0x1;
    int bitsWritten = 0;
    if(blockSize==0) blockSize=numBits;

    int numBlocks = 1;
    do{
      if(numBlocks==4) blockSize=numBits;
      int bitsToWrite=((blockSize>numBits-bitsWritten) ?
		       numBits-bitsWritten:blockSize);
      int count = 0;
      int lastBit;
      do{
	lastBit = (value & srcMask);
	if(lastBit!=0){
  	  buffer[nextDest] |= destMask;
	}
	destMask>>=1;
	if(destMask==0){
	  destMask=0x80;
	  nextDest++;
	  if(nextDest == buffer.length) growBuffer();
	  buffer[nextDest]=0;
	}
	srcMask<<=1;
      }
      while (bitsToWrite>++count);
      bitsWritten += bitsToWrite;
      if(bitsWritten<numBits){
	int tmpMask = srcMask;
	int i = bitsWritten;
	if(lastBit!=0){
	  do{
	    int nextBit = (value & tmpMask);
	    if (nextBit==0) break;
	    tmpMask <<= 1;
	  }
	  while(numBits>++i);
	}
	else{
	  do{
	    int nextBit = (value & tmpMask);
	    if(nextBit!=0) break;
	    tmpMask<<=1;
	  }
	  while(numBits>++i);
	}
	if(i<numBits){
          buffer[nextDest]|=destMask;
	}
	else bitsWritten = numBits;
	destMask>>=1;
	if(destMask== 0){
	  destMask=0x80;
	  nextDest++;
	  if(nextDest==buffer.length) growBuffer();
	  buffer[nextDest]=0;
	}
      }
      blockSize>>=1;
      if(blockSize<2) blockSize=2;
      numBlocks++;
    }
    while(numBits>bitsWritten);
  }
  void encodeCachedValue(int value, int numBits, IntCache cache){
    encodeCachedValue(value, numBits, cache, 0);
  }
  void encodeCachedValue(int value, int numBits, IntCache cache, int blockSize){
    // The next line is to avoid a warning;
    blockSize = 0;
    int newBlockSize = cache.getBlockSize(numBits);
    int[] index=new int[1];
    int[] vvalue=new int[1];
    boolean[] sameDiff=new boolean[1];

    vvalue[0]=value;
    if(cache.lookup(vvalue, index, 
		    Constants.PARTIAL_INT_MASK[numBits], sameDiff)){
      if(index[0]>1) index[0]++;
      for (int count = index[0]; count!=0; count--){
	destMask>>=1;
	if(destMask==0){
	  destMask=0x80;
	  nextDest++;
	  if(nextDest==buffer.length) growBuffer();
	  buffer[nextDest]=0;
	}
      }
      buffer[nextDest]|=destMask;
      destMask>>=1;
      if (destMask==0){
	destMask=0x80;
	nextDest++;
	if(nextDest==buffer.length) growBuffer();
	buffer[nextDest]=0;
      }
    }
    else{
      destMask>>=1;
      if(destMask==0){
	destMask=0x80;
	nextDest++;
	if(nextDest==buffer.length) growBuffer();
	buffer[nextDest]=0;
      }
      destMask>>=1;
      if(destMask==0){
	destMask=0x80;
	nextDest++;
	if(nextDest==buffer.length) growBuffer();
	buffer[nextDest]=0;
      }
      buffer[nextDest]|=destMask;
      destMask>>=1;
      if(destMask==0){
	destMask=0x80;
	nextDest++;
	if(nextDest==buffer.length) growBuffer();
	buffer[nextDest]=0;
      }
      if(sameDiff[0]){
        encodeValue(1, 1);
      }
      else {
	encodeValue(0, 1);
	encodeValue(vvalue[0], numBits, newBlockSize);
      }
    }
  }

  void encodeCachedValue(byte value, int numBits, CharCache cache){
    encodeCachedValue(value, numBits, cache, 0);
  }

  void encodeCachedValue(byte value, int numBits, CharCache cache, int blockSize){
    int[] index=new int[1];
    if(cache.lookup(value, index)){
      if (index[0]>1) index[0]++;
      for(int count=index[0]; count!=0; count--){
	destMask>>=1;
	if(destMask==0){
	  destMask=0x80;
	  nextDest++;
	  if(nextDest==buffer.length) growBuffer();
	  buffer[nextDest]=0;
	}
      }
      buffer[nextDest]|=destMask;
      destMask>>=1;
      if(destMask==0){
	destMask=0x80;
	nextDest++;
	if(nextDest==buffer.length) growBuffer();
	buffer[nextDest]=0;
      }
    }
    else{
      destMask>>=1;
      if(destMask==0){
	destMask=0x80;
	nextDest++;
	if(nextDest==buffer.length) growBuffer();
	buffer[nextDest]= 0;
      }
      destMask>>=1;
      if(destMask==0){
	destMask=0x80;
	nextDest++;
	if(nextDest==buffer.length) growBuffer();
	buffer[nextDest]=0;
      }
      buffer[nextDest]|=destMask;
      destMask>>=1;
      if(destMask==0){
	destMask=0x80;
	nextDest++;
	if(nextDest==buffer.length) growBuffer();
	buffer[nextDest]=(byte)0;
      }
      encodeValue(value, numBits, blockSize);
    }
  }

  void encodeCachedValue(int value, int numBits,
			 PixelCache cache, 
			 HuffmanCoder escapeCoder0,
			 HuffmanCoder escapeCoder1){
    int[] index=new int[1];
    if(cache.lookup(value, index)){
      if(index[0]>1) index[0]++;
      for(int count = index[0]; count!=0; count--){
	destMask>>=1;
	if(destMask==0){
	  destMask=0x80;
	  nextDest++;
	  if(nextDest==buffer.length) growBuffer();
	  buffer[nextDest]=0;
	}
      }
      buffer[nextDest]|=destMask;
      destMask>>=1;
      if(destMask==0){
	destMask=0x80;
	nextDest++;
	if(nextDest==buffer.length) growBuffer();
	buffer[nextDest]=0;
      }
    }
    else{
      // The value is not in the cache, so send an escape code, followed by
      // the value
      destMask>>=1;
      if(destMask==0){
	destMask=0x80;
	nextDest++;
	if(nextDest==buffer.length) growBuffer();
	buffer[nextDest]=0;
      }
      destMask>>=1;
      if(destMask==0){
	destMask=0x80;
	nextDest++;
	if(nextDest==buffer.length) growBuffer();
	buffer[nextDest]=0;
      }
      buffer[nextDest]|=destMask;
      destMask>>=1;
      if(destMask==0){
	destMask=0x80;
	nextDest++;
	if(nextDest==buffer.length) growBuffer();
	buffer[nextDest]=0;
      }
      // To transmit the value, use run-length coding with the static
      // Huffman code implemented by the supplied "escapeCoder" object
      //X encodeValue(value, numBits, numBits);
      int srcMask = 0x1;
      int pixelValue = ((value&srcMask)!=0 ? 1 : 0);
      encodeValue(pixelValue, 1, 1);
      for(int x = 0; x < numBits;){
	int runStart = x;
	if(pixelValue!=0){
	  while(x < numBits){
	    if((value & srcMask)==0) break;
	    srcMask <<= 1;
	    x++;
	  }
	}
	else{
	  while(x<numBits){
	    if((value & srcMask)!=0) break;
	    srcMask <<= 1;
	    x++;
	  }
	}
	int runLength = x - runStart;
	if(pixelValue!=0){
	  escapeCoder1.encode(runLength - 1, this);
	  pixelValue = 0;
	}
	else{
	  escapeCoder0.encode(runLength - 1, this);
	  pixelValue = 1;
	}
      }
    }
  }

  void encodeByte(byte value){
    if(destMask!=0x80){
      destMask=0x80;
      nextDest++;
      if(nextDest==buffer.length) growBuffer();
    }
    buffer[nextDest++]=value;
    if(nextDest==buffer.length) growBuffer();
  }

  byte[] getData(){
    return buffer;
  }
  int getDataLength(){
    int length = nextDest-PREFIX_SIZE;
    if (destMask!=0x80) length++;
    return length;
  }
  int getDataLengthInBits(){
    int length = nextDest-PREFIX_SIZE;
    length<<=3;
    int mask = destMask;
    while(mask!=0x80){
      length++;
      mask<<=1;
    }
    return length;
  }
  int getCumulativeBitsWritten(){
    int bitsWritten = ((nextDest-PREFIX_SIZE)<<3);
    int mask=0x80;
    while(mask!=destMask){
      mask>>=1;
      bitsWritten++;
    }
    int diff = bitsWritten-cumulativeBits;
    cumulativeBits=bitsWritten;
    return diff;
  }
  private void growBuffer(){
    int nextDestOffset = nextDest-PREFIX_SIZE;
    int newSize=size+size;
    byte[] newBuffer = new byte[newSize + PREFIX_SIZE];
    System.arraycopy(buffer, 0, newBuffer, 0, /*size*/buffer.length);
//    newBuffer[size]=0;
    buffer=newBuffer;
    size=newSize;
    nextDest=PREFIX_SIZE+nextDestOffset;
  }
}
