/* JDxpc -- DXPC in pure Java
 *
 *  Copyright (C) 2000 ymnk, JCraft, Inc.
 *
 *  Many thanks to 
 *    Brian Pane<brianp@cnet.com> and
 *    Zachary Vonler<lightborn@mail.utexas.edu>.
 *  JDxpc has been based on their awesome works, dxpc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package com.jcraft.jdxpc;

import java.io.*;
import java.net.*;
import java.lang.*;

class ClientChannel extends Channel{
  static private int idCount=5;
  synchronized static private int inc(){return idCount++;}

  private boolean firstRequest=true;
  private boolean firstReply=true;

  private int imageByteOrder;
  private int bitmapBitOrder;
  private int scanlineUnit;
  private int scanlinePad;

  InputStream clientInput=null;
  OutputStream clientOutput=null;
  Socket clientSocket=null;

  ClientReadBuffer rBuffer;

  ServerCache sCache=new ServerCache();
  SequenceNumQueue sNumQueue=new SequenceNumQueue();
  ClientCache cCache=new ClientCache();
  WriteBuffer wBuffer=new WriteBuffer();

  boolean bigEndian=false;

  private boolean running;

  int[] nextOpcode=new int[1];
  int[] start=new int[1];
  int[] length=new int[1];
  byte[] buff;

  int[][] requestData=new int[3][];

  int[] nextSequenceNum=new int[1];
  int[] dummySequenceNum=new int[1];
  int[] dummyOpcode=new int[1];

  ClientChannel(){
    super();
    setId(inc());
    requestData[0]=new int[1];
    requestData[1]=new int[1];
    requestData[2]=new int[1];
  }

  void setSocket(Socket socket){
    try	{
      clientSocket=socket;
      clientInput=socket.getInputStream();
      clientOutput=socket.getOutputStream();
      rBuffer=new ClientReadBuffer(this);
      rBuffer.setInputStream(clientInput);
    }
    catch(IOException e) {
      System.out.println("IO Error : " +  e );
    }
  }

  public synchronized void close(){
    // System.out.println("Closing Channel : " +  id );
    running = false;
    try{ 
      rBuffer.close ();
      clientSocket.close(); 
    }
    catch(Exception e){
      System.out.println ("Exception Channel : " + e);
    }
  }

  public void run(){
    running = true;
    EncodeBuffer eBuffer=new EncodeBuffer();
    while(running){
      try{ rBuffer.doRead(); }
      catch(java.io.IOException e){ 
        eBuffer.reset();
        proxy.proxyDrop(id, eBuffer);
        break;
      }
      doRead(rBuffer, eBuffer);
    }
  }

  private int[] rvalue=new int[1];

  void doRead(ReadBuffer rBuffer, EncodeBuffer eBuffer){
    int s;
    while((buff=rBuffer.getMessage(start, length))!=null){
      eBuffer.reset();
      s=start[0];

      if(firstRequest){
	for(int i=0; i<length[0]; i++){
	  eBuffer.encodeValue((int)buff[s+i], 8);
	}
	proxy.proxyWrite(id, eBuffer);
	firstRequest=false;
	continue;
      }

      cCache.lastRequestSequenceNum++;
      if((cCache.lastRequestSequenceNum&~0xffff)!=0){
        cCache.lastRequestSequenceNum&=0xffff;
      }
      int opcode=(int)(buff[s]&0xff);
      if((opcode==70 /*X_PolyFillRectangle*/) &&
	 rBuffer.getUINT(2)==3){
	opcode=127;
      }
      eBuffer.encodeCachedValue((byte)opcode, 8, 
				cCache.opcodeCache[cCache.lastOpcode]);
      cCache.lastOpcode=opcode;

//System.out.println("opcode: "+opcode);

      switch(opcode){
      case 84: // X_AllocColor:
	{
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4), 
				    29,
				    cCache.colormapCache, 9);
	  int nextSrc=8;
	  int[] colorData=new int[3];
	  for(int i=0; i<3; i++){
	    int value=rBuffer.getUINT(nextSrc);
	    eBuffer.encodeCachedValue(value, 16,
				      cCache.allocColorRGBCache[i], 4);
	    colorData[i]=value;
	    nextSrc+=2;
	  }
	  sNumQueue.push(cCache.lastRequestSequenceNum, opcode,
			 colorData[0], colorData[1], colorData[2]);
	}
	break;
      case 18: // X_ChangeProperty:
	{
	  byte format=buff[s+16];
	  eBuffer.encodeCachedValue(format, 8,
				    cCache.changePropertyFormatCache);
	  int dataLength=rBuffer.getULONG(20);
	  eBuffer.encodeValue(dataLength, 32, 6);
	  eBuffer.encodeValue(buff[s+1], 2);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.windowCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				    29, cCache.changePropertyPropertyCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(12),
				    29, cCache.changePropertyTypeCache, 9);
	  int nextSrc=24;
	  if(format==8){
	    nextSrc+=s;
	    cCache.changePropertyTextCompressor.reset();
	    for(int i=0; i<dataLength; i++){
	      cCache.changePropertyTextCompressor.encodeChar(buff[nextSrc++], eBuffer);
	    }
	  }
	  else if(format==32){
	    for(int i=0; i < dataLength; i++){
	      eBuffer.encodeCachedValue(rBuffer.getULONG(nextSrc),
					32, cCache.changePropertyData32Cache);
	      nextSrc+=4;
	    }
	  }
	  else{
	    for(int i=0; i<dataLength; i++){
	      eBuffer.encodeValue(rBuffer.getUINT(nextSrc), 16);
	      nextSrc+=2;
	    }
	  }
	}
	break;
      case 2: // X_ChangeWindowAttributes:
	{
	  eBuffer.encodeValue((length[0]-12)>>>2, 4);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.windowCache, 9);
	  int bitmask=rBuffer.getULONG(8);
	  eBuffer.encodeCachedValue(bitmask, 15,
				    cCache.createWindowBitmaskCache);
	  int nextSrc=12;
	  int mask=0x1;
	  for(int j=0; j<15; j++){
	    if((bitmask&mask)!=0){
	      eBuffer.encodeCachedValue(rBuffer.getULONG(nextSrc),
					32, cCache.createWindowAttrCache[j]);
	      nextSrc+=4;
	    }
	    mask<<=1;
	  }
	}
	break;
      case 61: // X_ClearArea:
	{
	  eBuffer.encodeValue((int)buff[s+1], 1);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.windowCache, 9);
	  int nextSrc=8;
	  for(int i=0; i<4; i++){
	    eBuffer.encodeCachedValue(rBuffer.getUINT(nextSrc),
				      16, cCache.clearAreaGeomCache[i], 8);
	    nextSrc += 2;
	  }
	}
	break;
      case 46: // X_CloseFont:
	{
	  int font=rBuffer.getULONG(4);
	  eBuffer.encodeValue(font-cCache.lastFont, 29, 5);
	  cCache.lastFont=font;
	}
	break;
      case 12: // X_ConfigureWindow:
	{
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.windowCache, 9);
	  int bitmask=rBuffer.getUINT(8);
	  eBuffer.encodeCachedValue(bitmask, 7,
				    cCache.configureWindowBitmaskCache);
	  int mask=0x1;
	  int nextSrc=12;
	  for(int i=0; i<7; i++){
	    if((bitmask&mask)!=0){
	      eBuffer.encodeCachedValue(rBuffer.getULONG(nextSrc),
					Constants.CONFIGUREWINDOW_FIELD_WIDTH[i],
					cCache.configureWindowAttrCache[i], 8);
	      nextSrc+=4;
	    }
	    mask<<=1;
	  }
	}
	break;
      case 24: // X_ConvertSelection:
	{
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29,	cCache.convertSelectionRequestorCache, 9);
	  int nextSrc=8;
	  for(int i=0; i<3; i++){
	    eBuffer.encodeCachedValue(rBuffer.getULONG(nextSrc), 29,
				      cCache.convertSelectionAtomCache[i], 9);
	    nextSrc+=4;
	  }
	  int timestamp=rBuffer.getULONG(nextSrc);
	  eBuffer.encodeValue(timestamp-cCache.convertSelectionLastTimestamp, 32, 4);
	  cCache.convertSelectionLastTimestamp=timestamp;
	}
	break;
      case 62: // X_CopyArea:
	{
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.drawableCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				    29, cCache.drawableCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(12),
				    29, cCache.gcCache, 9);
	  int nextSrc=16;
	  for(int i=0; i<6; i++){
	    eBuffer.encodeCachedValue(rBuffer.getUINT(nextSrc),
				      16, cCache.copyAreaGeomCache[i], 8);
	    nextSrc+=2;
	  }
	}
	break;
      case 57: // X_CopyGC:
	{
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.gcCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				    29, cCache.gcCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(12),
				    23, cCache.createGCBitmaskCache);
	}
	break;
      case 63: // X_CopyPlane:
	{
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.drawableCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				    29, cCache.drawableCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(12),
				    29, cCache.gcCache, 9);
	  int nextSrc=16;
	  for(int i=0; i<6; i++){
	    eBuffer.encodeCachedValue(rBuffer.getUINT(nextSrc),
				      16, cCache.copyPlaneGeomCache[i], 8);
	    nextSrc+=2;
	  }
	  eBuffer.encodeCachedValue(rBuffer.getULONG(28),
				    32, cCache.copyPlaneBitPlaneCache, 10);
	}
	break;
      case 55: // X_CreateGC:
      case 56: // X_ChangeGC:
	{
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4), 
				    29, cCache.gcCache, 9);
	  int nextSrc=/*buffer+*/8;
	  if (opcode==55/*X_CreateGC*/){
	    eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				      29, cCache.drawableCache, 9);
	    nextSrc+=4;
	  }
	  int bitmask=rBuffer.getULONG(nextSrc);
	  nextSrc+=4;
	  eBuffer.encodeCachedValue(bitmask, 23,
				    cCache.createGCBitmaskCache);
	  int mask = 0x1;
	  for(int i=0; i<23; i++){
	    if((bitmask & mask)!=0){
	      int value=rBuffer.getULONG(nextSrc);
	      nextSrc+=4;
	      int fieldWidth=Constants.CREATEGC_FIELD_WIDTH[i];
	      if(fieldWidth<=4){
		eBuffer.encodeValue(value, fieldWidth);
	      }
	      else{
		eBuffer.encodeCachedValue(value, fieldWidth,
					  cCache.createGCAttrCache[i]);
	      }
	    }
	    mask<<=1;
	  }
	}
	break;
      case 53: // X_CreatePixmap:
	{
	  eBuffer.encodeCachedValue(buff[s+1], 8,
					 cCache.depthCache);
	  int pixmap=rBuffer.getULONG(4);
	  int diff=pixmap-cCache.createPixmapLastPixmap;
	  if(diff==0){
	    eBuffer.encodeValue(1, 1);
	  }
	  else{
	    eBuffer.encodeValue(0, 1);
	    eBuffer.encodeValue(diff, 29, 4);
	    cCache.createPixmapLastPixmap = pixmap;
	  }
	  eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				    29, cCache.drawableCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getUINT(12),
				    16, cCache.createPixmapXCache, 8);
	  eBuffer.encodeCachedValue(rBuffer.getUINT(14),
				    16, cCache.createPixmapYCache, 8);
	}
	break;
      case 1: // X_CreateWindow:
	{
	  eBuffer.encodeCachedValue(buff[s+1], 8, cCache.depthCache);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				    29, cCache.windowCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.windowCache, 9);
	  int nextSrc=12;
	  for(int i=0; i<6; i++){
	    eBuffer.encodeCachedValue(rBuffer.getUINT(nextSrc),
				      16, cCache.createWindowGeomCache[i], 8);
	    nextSrc+=2;
	  }
	  eBuffer.encodeCachedValue(rBuffer.getULONG(24), 29, cCache.visualCache);
	  int bitmask=rBuffer.getULONG(28);
	  eBuffer.encodeCachedValue(bitmask, 15,
				    cCache.createWindowBitmaskCache);
	  nextSrc=32;
	  int mask=0x1;
	  for(int j=0; j<15; j++){
	    if((bitmask & mask)!=0){
	      eBuffer.encodeCachedValue(rBuffer.getULONG(nextSrc),
					32, cCache.createWindowAttrCache[j]);
	      nextSrc+=4;
	    }
	    mask<<=1;
	  }
	}
	break;
      case 19: // X_DeleteProperty:
	{
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4), 29,
				    cCache.windowCache, 9);
	  eBuffer.encodeValue(rBuffer.getULONG(8), 29, 9);
	}
	break;
      case 69: // X_FillPoly:
	{
	  int numPoints=((length[0]-16)>>>2);
	  eBuffer.encodeCachedValue(numPoints, 14,
				    cCache.fillPolyNumPointsCache, 4);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.drawableCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				    29, cCache.gcCache, 9);
	  eBuffer.encodeValue((int)buff[s+12], 2);
	  eBuffer.encodeValue((int)buff[s+13], 1);
	  boolean relativeCoordMode=(buff[s+13]!=0);
	  int nextSrc=16;
	  int pointIndex = 0;
	  for(int i=0; i<numPoints; i++){
	    if(relativeCoordMode){
	      eBuffer.encodeCachedValue(rBuffer.getUINT(nextSrc), 16,
					cCache.fillPolyXRelCache[pointIndex], 8);
	      nextSrc+=2;
	      eBuffer.encodeCachedValue(rBuffer.getUINT(nextSrc), 16,
					cCache.fillPolyYRelCache[pointIndex], 8);
	      nextSrc+=2;
	    }
	    else{
	      int x=rBuffer.getUINT(nextSrc);
	      nextSrc+=2;
	      int y=rBuffer.getUINT(nextSrc);
	      nextSrc+=2;
	      int j;
	      for(j=0; j<8; j++){
		if((x==cCache.fillPolyRecentX[j]) &&
		   (y==cCache.fillPolyRecentY[j]))
		  break;
	      }
	      if(j<8){
		eBuffer.encodeValue(1, 1);
		eBuffer.encodeValue(j, 3);
	      }
	      else{
		eBuffer.encodeValue(0, 1);
		eBuffer.encodeCachedValue(x, 16,
					  cCache.fillPolyXAbsCache[pointIndex], 8);
		eBuffer.encodeCachedValue(y, 16,
					  cCache.fillPolyYAbsCache[pointIndex], 8);
		cCache.fillPolyRecentX[cCache.fillPolyIndex]=x;
		cCache.fillPolyRecentY[cCache.fillPolyIndex]=y;
		cCache.fillPolyIndex++;
		if(cCache.fillPolyIndex==8){
		  cCache.fillPolyIndex=0;
		}
	      }
	    }
	    if(pointIndex+1<Constants.FILL_POLY_MAX_POINTS){
	      pointIndex++;
	    }
	  }
	}
	break;
      case 88: // X_FreeColors:
	{
	  int numPixels=rBuffer.getUINT(2);
	  eBuffer.encodeValue(numPixels, 16, 4);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.colormapCache, 9);
	  eBuffer.encodeValue(rBuffer.getULONG(8), 32, 4);
	  int nextSrc=12;
	  while(numPixels>0){
	    eBuffer.encodeValue(rBuffer.getULONG(nextSrc), 32, 8);
	    nextSrc+=4;
	    numPixels--;
	  }
	}
	break;
      case 95: //X_FreeCursor:
	{
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.cursorCache, 9);
	}
	break;
      case 60: // X_FreeGC:
	{
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.gcCache, 9);
	}
	break;
      case 54: // X_FreePixmap:
	{
	  int pixmap=rBuffer.getULONG(4);
	  int diff=pixmap-cCache.createPixmapLastPixmap;
	  if(diff==0){
	    eBuffer.encodeValue(1, 1);
	  }
	  else{
	    eBuffer.encodeValue(0, 1);
	    cCache.createPixmapLastPixmap = pixmap;
	    eBuffer.encodeValue(diff, 29, 4);
	  }
	}
	break;
      case 17: // X_GetAtomName:
	{
	  eBuffer.encodeValue(rBuffer.getULONG(4), 29, 9);
	  sNumQueue.push(cCache.lastRequestSequenceNum, opcode);
	}
	break;
      case 14: X_GetGeometry:
	{
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.drawableCache, 9);
	  sNumQueue.push(cCache.lastRequestSequenceNum, opcode);
	}
	break;
      case 43: // X_GetInputFocus:
      case 119: // X_GetModifierMapping:
	{
	  sNumQueue.push(cCache.lastRequestSequenceNum, opcode);
	}
	break;
      case 101: // X_GetKeyboardMapping:
	{
	  eBuffer.encodeValue((int)buff[s+4], 8);
	  eBuffer.encodeValue((int)buff[s+5], 8);
	  sNumQueue.push(cCache.lastRequestSequenceNum, opcode);
	}
	break;
      case 20: // X_GetProperty:
	{
	  eBuffer.encodeValue((int)buff[s+1], 1);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4), 29,
				    cCache.windowCache, 9);
	  int property=rBuffer.getULONG(8);
	  eBuffer.encodeValue(property, 29, 9);
	  eBuffer.encodeValue(rBuffer.getULONG(12), 29, 9);
	  eBuffer.encodeValue(rBuffer.getULONG(16), 32, 2);
	  eBuffer.encodeValue(rBuffer.getULONG(20), 32, 8);
	  sNumQueue.push(cCache.lastRequestSequenceNum, opcode, property);
	}
	break;
      case 23: // X_GetSelectionOwner:
	{
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29,
				    cCache.getSelectionOwnerSelectionCache, 9);
	  sNumQueue.push(cCache.lastRequestSequenceNum, opcode);
	}
	break;
      case 28: // X_GrabButton:
      case 26: // X_GrabPointer:
	{
	  eBuffer.encodeValue((int)buff[s+1], 1);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.windowCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getUINT(8),
				    16, cCache.grabButtonEventMaskCache);
	  eBuffer.encodeValue((int)buff[s+10], 1);
	  eBuffer.encodeValue((int)buff[s+11], 1);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(12),
				    29, cCache.grabButtonConfineCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(16),
				    29, cCache.cursorCache, 9);
	  if(opcode==28/*X_GrabButton*/){
	    eBuffer.encodeCachedValue(buff[s+20], 8,
				      cCache.grabButtonButtonCache);
	    eBuffer.encodeCachedValue(rBuffer.getUINT(22),
				      16, cCache.grabButtonModifierCache);
	  }
	  else{
	    int timestamp=rBuffer.getULONG(20);
	    eBuffer.encodeValue(timestamp-cCache.grabKeyboardLastTimestamp, 32, 4);
	    cCache.grabKeyboardLastTimestamp=timestamp;
	    sNumQueue.push(cCache.lastRequestSequenceNum, opcode);
	  }
	}
	break;
      case 31: // X_GrabKeyboard:
	{
	  eBuffer.encodeValue((int)buff[s+1], 1);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.windowCache, 9);
	  int timestamp=rBuffer.getULONG(8);
	  eBuffer.encodeValue(timestamp-cCache.grabKeyboardLastTimestamp, 32, 4);
	  cCache.grabKeyboardLastTimestamp=timestamp;
	  eBuffer.encodeValue((int)buff[s+12], 1);
	  eBuffer.encodeValue((int)buff[s+13], 1);
	  sNumQueue.push(cCache.lastRequestSequenceNum, opcode);
	}
	break;
      case 36: // X_GrabServer:
      case 37: // X_UngrabServer:
      case 127: // X_NoOperation:
	{
	}
	break;
      case 76: // X_ImageText8:
	{
	  int textLength=(int)buff[s+1];
	  eBuffer.encodeValue(textLength, 8);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.drawableCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				    29, cCache.gcCache, 9);
	  int x=rBuffer.getUINT(12);
	  int xDiff=x-cCache.imageText8LastX;
	  cCache.imageText8LastX=x;
	  eBuffer.encodeCachedValue(xDiff, 16,
				    cCache.imageText8CacheX, 8);
	  int y=rBuffer.getUINT(14);
	  int yDiff=y-cCache.imageText8LastY;
	  cCache.imageText8LastY = y;
	  eBuffer.encodeCachedValue(yDiff, 16,
				    cCache.imageText8CacheY, 8);
	  int nextSrc=s+16;
	  cCache.imageText8TextCompressor.reset();
	  for(int j=0; j<textLength; j++){
	    cCache.imageText8TextCompressor.encodeChar(buff[nextSrc++], eBuffer);
	  }
	}
	break;
      case 16: // X_InternAtom:
	{
	  int nameLength=rBuffer.getUINT(4);
	  eBuffer.encodeValue(nameLength, 16, 6);
	  eBuffer.encodeValue((int)buff[s+1], 1);
	  int nextSrc=s+8;
	  cCache.internAtomTextCompressor.reset();
	  for(int i=0; i<nameLength; i++){
	    cCache.internAtomTextCompressor.encodeChar(buff[nextSrc++], eBuffer);
	  }
	  sNumQueue.push(cCache.lastRequestSequenceNum, opcode);
	}
	break;
      case 99: // X_ListExtensions:
	{
	  sNumQueue.push(cCache.lastRequestSequenceNum, opcode);
	}
	break;
      case 49: // X_ListFonts:
	{
	  int textLength=rBuffer.getUINT(6);
	  eBuffer.encodeValue(textLength, 16, 6);
	  eBuffer.encodeValue(rBuffer.getUINT(4), 16, 6);
	  int nextSrc=s+8;
	  cCache.polyText8TextCompressor.reset();
	  for(int i=0; i<textLength; i++){
	    cCache.polyText8TextCompressor.encodeChar(buff[nextSrc++], eBuffer);
	  }
	  sNumQueue.push(cCache.lastRequestSequenceNum, opcode);
	}
	break;
      case 92: // X_LookupColor:
      case 85: // X_AllocNamedColor:
	{
	  sNumQueue.push(cCache.lastRequestSequenceNum, opcode);
	  int textLength=rBuffer.getUINT(8);
	  eBuffer.encodeValue(textLength, 16, 6);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.colormapCache, 9);
	  int nextSrc=s+12;
	  cCache.polyText8TextCompressor.reset();
	  for(int i=0; i<textLength; i++){
	    cCache.polyText8TextCompressor.encodeChar(buff[nextSrc++], eBuffer);
	  }
	}
	break;
      case  8: // X_MapWindow:
      case 10: // X_UnmapWindow:
      case  9: // X_MapSubwindows:
      case  3: // X_GetWindowAttributes:
      case  4: // X_DestroyWindow:
      case  5: // X_DestroySubwindows:
      case 38: // X_QueryPointer:
      case 15: // X_QueryTree:
	{
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.windowCache, 9);
	  if((opcode==38/*X_QueryPointer*/)||
	     (opcode==3/*X_GetWindowAttributes*/)||
	     (opcode==15/*X_QueryTree*/)){
	    sNumQueue.push(cCache.lastRequestSequenceNum, opcode);
	  }
	}
	break;
      case 45: // X_OpenFont:
	{
	  int nameLength=rBuffer.getUINT(8);
	  eBuffer.encodeValue(nameLength, 16, 7);
	  int font=rBuffer.getULONG(4);
	  eBuffer.encodeValue(font-cCache.lastFont, 29, 5);
	  cCache.lastFont=font;
	  int nextSrc=s+12;
	  cCache.openFontTextCompressor.reset();
	  for(;nameLength>0; nameLength--){
	    cCache.openFontTextCompressor.encodeChar(buff[nextSrc++], eBuffer);
	  }
	}
	break;
      case 70: // X_PolyFillRectangle:
	{
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.drawableCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				    29, cCache.gcCache, 9);
	  int index=0;
	  int lastX=0, lastY=0;
	  int lastWidth=0, lastHeight=0;
	  for(int i=12; i<length[0];){
	    int x=rBuffer.getUINT(i);
	    int newX=x;
	    x-=lastX;
	    lastX=newX;
	    eBuffer.encodeCachedValue(x, 16,
				      cCache.polyFillRectangleCacheX[index], 8);
	    i+=2;
	    int y=rBuffer.getUINT(i);
	    int newY=y;
	    y-=lastY;
	    lastY=newY;
	    eBuffer.encodeCachedValue(y, 16,
				      cCache.polyFillRectangleCacheY[index], 8);
	    i+=2;
	    int width=rBuffer.getUINT(i);
	    int newWidth=width;
	    width-=lastWidth;
	    lastWidth=newWidth;
	    eBuffer.encodeCachedValue(width, 16,
				      cCache.polyFillRectangleCacheWidth[index], 8);
	    i+=2;
	    int height=rBuffer.getUINT(i);
	    int newHeight=height;
	    height-=lastHeight;
	    lastHeight=newHeight;
	    eBuffer.encodeCachedValue(height, 16,
				      cCache.polyFillRectangleCacheHeight[index], 8);
	    i+=2;
	    index=1;
	    eBuffer.encodeValue(((i<length[0])?1:0), 1);
	  }
	}
	break;
      case 64: // X_PolyPoint:
	{
	  eBuffer.encodeValue(rBuffer.getUINT(2)-3, 16, 4);
	  eBuffer.encodeValue((int)buff[s+1], 1);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.drawableCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				    29, cCache.gcCache, 9);
	  int nextSrc=12;
	  int index=0;
	  int lastX=0, lastY=0;
	  for(int i=12; i<length[0]; i+=4){
	    int x=rBuffer.getUINT(nextSrc);
	    nextSrc+=2;
	    int tmp=x;
	    x-=lastX;
	    lastX=tmp;
	    eBuffer.encodeCachedValue(x, 16, cCache.polyPointCacheX[index], 8);
	    int y=rBuffer.getUINT(nextSrc);
	    nextSrc+=2;
	    tmp=y;
	    y-=lastY;
	    lastY=tmp;
	    eBuffer.encodeCachedValue(y, 16, cCache.polyPointCacheY[index], 8);
	    index=1;
	  }
	}
	break;
      case 65: // X_PolyLine:
	{
	  eBuffer.encodeValue(rBuffer.getUINT(2)-3, 16, 4);
	  eBuffer.encodeValue((int)buff[s+1], 1);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4), 29, cCache.drawableCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(8), 29, cCache.gcCache, 9);
	  int nextSrc=12;
	  int index=0;
	  int lastX=0, lastY=0;
	  for(int i=12; i<length[0]; i+=4){
	    int x=rBuffer.getUINT(nextSrc);
	    nextSrc+=2;
	    int tmp=x;
	    x-=lastX;
	    lastX=tmp;
	    eBuffer.encodeCachedValue(x, 16, cCache.polyLineCacheX[index], 8);
	    int y=rBuffer.getUINT(nextSrc);
	    nextSrc+=2;
	    tmp=y;
	    y-=lastY;
	    lastY=tmp;
	    eBuffer.encodeCachedValue(y, 16, cCache.polyLineCacheY[index], 8);
	    index=1;
	  }
	}
	break;
      case 67: // X_PolyRectangle:
	{
	  eBuffer.encodeValue((rBuffer.getUINT(2)-3)>>>1, 16, 3);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.drawableCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				    29, cCache.gcCache, 9);
	  int end=length[0];
	  int nextSrc=12;
	  while(nextSrc<end){
	    for(int i=0; i<4; i++){
	      eBuffer.encodeCachedValue(rBuffer.getUINT(nextSrc), 16,
					cCache.polyRectangleGeomCache[i], 8);
	      nextSrc+=2;
	    }
	  }
	}
	break;
      case 66: // X_PolySegment:
	{
	  eBuffer.encodeValue((rBuffer.getUINT(2)-3)>>>1, 16, 4);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4), 
				    29, cCache.drawableCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				    29, cCache.gcCache, 9);
	  int end=length[0];
	  int nextSrc=12;
	  while(nextSrc<end){
	    int x=rBuffer.getUINT(nextSrc);
	    nextSrc+=2;
	    int xDiff0=x-cCache.polySegmentLastX[0];
	    int xDiff1=x-cCache.polySegmentLastX[1];
	    int xDiff0Abs=(int)xDiff0;
	    if(xDiff0Abs<0) xDiff0Abs=-xDiff0Abs;
	    int xDiff1Abs=(int)xDiff1;
	    if(xDiff1Abs<0) xDiff1Abs=-xDiff1Abs;

	    int y=rBuffer.getUINT(nextSrc);
	    nextSrc+=2;
	    int yDiff0=y-cCache.polySegmentLastY[0];
	    int yDiff1=y-cCache.polySegmentLastY[1];
	    int yDiff0Abs=(int)yDiff0;
	    if(yDiff0Abs<0) yDiff0Abs=-yDiff0Abs;
	    int yDiff1Abs=(int) yDiff1;
	    if(yDiff1Abs<0) yDiff1Abs=-yDiff1Abs;

	    int diff0=xDiff0Abs+yDiff0Abs;
	    int diff1=xDiff1Abs+yDiff1Abs;
	    if(diff0<diff1){
	      eBuffer.encodeValue(0, 1);
	      eBuffer.encodeCachedValue(xDiff0, 16,
					cCache.polySegmentCacheX, 6);
	      eBuffer.encodeCachedValue(yDiff0, 16,
					cCache.polySegmentCacheY, 6);
	    }
	    else{
	      eBuffer.encodeValue(1, 1);
	      eBuffer.encodeCachedValue(xDiff1, 16,
					cCache.polySegmentCacheX, 6);
	      eBuffer.encodeCachedValue(yDiff1, 16,
					cCache.polySegmentCacheY, 6);
	    }

	    cCache.polySegmentLastX[cCache.polySegmentCacheIndex]=x;
	    cCache.polySegmentLastY[cCache.polySegmentCacheIndex]=y;
	    cCache.polySegmentCacheIndex=
	      (cCache.polySegmentCacheIndex==1?0:1);
	  }
	}
	break;
      case 74: // X_PolyText8:
	{
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.drawableCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				    29, cCache.gcCache, 9);
	  int x=rBuffer.getUINT(12);
	  int xDiff=x-cCache.polyText8LastX;
	  cCache.polyText8LastX=x;
	  eBuffer.encodeCachedValue(xDiff, 16,
				    cCache.polyText8CacheX, 8);
	  int y=rBuffer.getUINT(14);
	  int yDiff=y-cCache.polyText8LastY;
	  cCache.polyText8LastY=y;
	  eBuffer.encodeCachedValue(yDiff, 16,
				    cCache.polyText8CacheY, 8);
	  int end=s+length[0]-1;
	  int nextSrc=s+16;
	  while(nextSrc<end){
	    int textLength=(int)buff[nextSrc++];
	    eBuffer.encodeValue(1, 1);
	    eBuffer.encodeValue(textLength, 8);
	    if(textLength==255){
	      eBuffer.encodeCachedValue(rBuffer.getULONG(nextSrc-s, true),
					29, cCache.polyText8FontCache);
	      nextSrc+=4;
	    }
	    else{
	      eBuffer.encodeCachedValue(buff[nextSrc++],
					8, cCache.polyText8DeltaCache);
	      cCache.polyText8TextCompressor.reset();
	      for(int i=0; i<textLength; i++){
		cCache.polyText8TextCompressor.encodeChar(buff[nextSrc++], eBuffer);
	      }
	    }
	  }
	  eBuffer.encodeValue(0, 1);
	}
	break;
      case 72: // X_PutImage:
	{
	  eBuffer.encodeValue(rBuffer.getUINT(2), 16, 8);
	  eBuffer.encodeValue((int)buff[s+1], 2);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.drawableCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(8),
				    29, cCache.gcCache, 9);
	  int width = rBuffer.getUINT(12);
	  eBuffer.encodeCachedValue(width, 16,
				    cCache.putImageWidthCache, 8);
	  int height = rBuffer.getUINT(14);
	  eBuffer.encodeCachedValue(height, 16,
				    cCache.putImageHeightCache, 8);
	  int x=rBuffer.getUINT(16);
	  int xDiff=x-cCache.putImageLastX;
	  cCache.putImageLastX=x;
	  eBuffer.encodeCachedValue(xDiff, 16,
				    cCache.putImageXCache, 8);
	  int y=rBuffer.getUINT(18);
	  int yDiff=y-cCache.putImageLastY;
	  cCache.putImageLastY=y;
	  eBuffer.encodeCachedValue(yDiff, 16,
				    cCache.putImageYCache, 8);
	  eBuffer.encodeCachedValue(buff[s+20], 8,
				    cCache.putImageOffsetCache);
	  eBuffer.encodeCachedValue(buff[s+21], 8,
				    cCache.depthCache);
	  int nextSrc=s+24;
          if((buff[s+1]==0) &&(height<=32)&&
	     (width>height*Constants.PUT_IMAGE_MIN_ASPECT_RATIO)){
	    if((imageByteOrder==0) && (bitmapBitOrder==0)){
	      int next=s+24;
	      for(int i=24; i<length[0]; i++){
		buff[next]=Constants.REVERSED_BYTE[buff[next]];
		next++;
	      }
	    }
	    int widthInBits=((width/scanlinePad)*scanlinePad);
	    if(widthInBits<width) widthInBits+=scanlinePad;
	    int widthInBytes=(widthInBits>>>3);
	    int nextSrc2=s+24;
	    int srcMask=0x80;
	    cCache.putImageLastPixels.reset();
	    for(int xCoord=0; xCoord<width; xCoord++){
	      int columnValue=0;
	      int next=nextSrc2;
	      for(int h=0; h<height; h++){
		columnValue<<=1;
		if((srcMask&buff[next])!=0){
		  columnValue|=1;
		}
		next+=widthInBytes;
	      }
	      int modelNum=cCache.putImageLastPixels.getValue();
	      eBuffer.encodeCachedValue(columnValue, height,
					cCache.putImagePixelCache[modelNum %
						ClientCache.PUT_IMAGE_PIXEL_CACHE_SIZE],
					cCache.columnPixel0Coder,
					cCache.columnPixel1Coder);
	      cCache.putImageLastPixels.add(columnValue);
	      srcMask>>>=1;
	      if(srcMask==0){
		srcMask=0x80;
		nextSrc2++;
	      }
	    }
	  }
	  else if(buff[s+1]==0){
	    if(width+2>cCache.putImageLineSize){
	      cCache.putImageLineSize=width+2;
	      cCache.putImageReferenceLine= new int[width + 2];
	      cCache.putImageCodingLine= new int[width + 2];
	    }
	    int widthInBits=((width/scanlinePad)*scanlinePad);
	    if(widthInBits<width) widthInBits+=scanlinePad;
	    int widthInBytes=(widthInBits>>>3);
	    for(int h=0; h<height; h++){
	      int codingLineLength=0;
	      int nextSrc2=s+24+h*widthInBytes;
	      byte nextSrcChar=buff[nextSrc2];
	      if(h!=0) nextSrcChar^=buff[nextSrc2-widthInBytes];
	      if((imageByteOrder==0)&&(bitmapBitOrder==0))
		nextSrcChar=Constants.REVERSED_BYTE[(nextSrcChar&0xff)];

	      int srcMask=0x80;
	      int lastPixelValue=0;
	      int pixelValue=((nextSrcChar&srcMask)!=0 ? 1 : 0);
	      if(h==0){
		eBuffer.encodeValue(pixelValue, 1);
	      }
	      for(int xCoord=0; xCoord<width;){
		int runStart=xCoord;
		if(pixelValue!=0){
		  if(pixelValue!=lastPixelValue){
		    cCache.putImageCodingLine[codingLineLength++]=xCoord;
		  }
		  while(xCoord<width){
		    if((nextSrcChar&srcMask)==0){
		      break;
		    }
		    srcMask>>>=1;
		    if(srcMask==0){
		      srcMask=0x80;
		      nextSrc2++;
		      if(xCoord+1<width){
			nextSrcChar=buff[nextSrc2];
			if(h!=0){
			  nextSrcChar^=buff[nextSrc2-widthInBytes];
			}
			if((imageByteOrder==0) &&
			   (bitmapBitOrder==0))
			  nextSrcChar=Constants.REVERSED_BYTE[(nextSrcChar&0xff)];
		      }
		    }
		    xCoord++;
		  }
		  lastPixelValue=pixelValue;
		}
		else{
		  if(pixelValue != lastPixelValue){
		    cCache.putImageCodingLine[codingLineLength++] = xCoord;
		  }
		  while(xCoord<width){
		    if((nextSrcChar&srcMask)!=0){
		      break;
		    }
		    srcMask>>>=1;
		    if(srcMask==0){
		      srcMask=0x80;
		      nextSrc2++;
		      if(xCoord+1<width){
			nextSrcChar=buff[nextSrc2];
			if(h!=0){
			  nextSrcChar^= buff[nextSrc2-widthInBytes];
			}
			if((imageByteOrder==0) && (bitmapBitOrder==0)){
			  nextSrcChar=Constants.REVERSED_BYTE[(nextSrcChar&0xff)];
			}
		      }
		    }
		    xCoord++;
		  }
		  lastPixelValue=pixelValue;
		}
		// here 'nextSrc' points to either a color change or the
		// pixel after the end of the scan line.  Thus the length
		// of the solid-color block that just ended can be encoded
		// (invariant: runLength >= 1)
		int runLength=xCoord-runStart;
		if(pixelValue!=0){
		  if(h==0){
		    cCache.putImagePixel1Coder.encode(runLength, eBuffer);
		  }
		  pixelValue=0;
		}
		else{
		  if(h==0){
		    cCache.putImagePixel0Coder.encode(runLength, eBuffer);
		  }
		  pixelValue=1;
		}
	      }
// ??
	      cCache.putImageCodingLine[codingLineLength++]=width;
	      cCache.putImageCodingLine[codingLineLength++]=width;

	      if(h!=0){
		int lastX=0;
		int nextCodingIndex=0, nextReferenceIndex=0;
		while(lastX<width){
		  int nextCoding=cCache.putImageCodingLine[nextCodingIndex];
		  int nextReference=cCache.putImageReferenceLine[nextReferenceIndex];
		  if(nextCoding==nextReference){
		    cCache.putImageDiffCoder.encode(ClientCache.SD_VERTICAL_0,
						    eBuffer);
		    lastX=nextCoding;
		    nextCodingIndex++;
		    nextReferenceIndex++;
		    continue;
		  }
		  if(nextCoding>nextReference){
		    int diff=nextCoding-nextReference;
		    if(diff==1){
		      cCache.putImageDiffCoder.encode(ClientCache.SD_VERTICAL_PLUS_1,
							    eBuffer);
		      lastX=nextCoding;
		      nextCodingIndex++;
		      nextReferenceIndex++;
		    }
		    else if(diff==2){
		      cCache.putImageDiffCoder.encode(ClientCache.SD_VERTICAL_PLUS_2,  eBuffer);
		      lastX=nextCoding;
		      nextCodingIndex++;
		      nextReferenceIndex++;
		    }
		    else{
		      cCache.putImageDiffCoder.encode(ClientCache.SD_PASS, eBuffer);
		      nextReferenceIndex+=2;
		    }
		  }
		  else{		// (nextCoding < nextReference)
		    int diff=nextReference-nextCoding;
		    if(nextReference==width) diff=99999;
		    if(diff == 1){
		      cCache.putImageDiffCoder.encode(ClientCache.SD_VERTICAL_MINUS_1, eBuffer);
		      lastX=nextCoding;
		      nextCodingIndex++;
		      nextReferenceIndex++;
		    }
		    else if(diff==2){
		      cCache.putImageDiffCoder.encode(ClientCache.SD_VERTICAL_MINUS_2, eBuffer);
		      lastX=nextCoding;
		      nextCodingIndex++;
		      nextReferenceIndex++;
		    }
		    else{
		      cCache.putImageDiffCoder.encode(ClientCache.SD_HORIZONTAL, eBuffer);
		      if((nextCodingIndex & 1)!=0){
			cCache.putImagePixel0Coder.encode(nextCoding-lastX, eBuffer);
		      }
		      else{
			cCache.putImagePixel1Coder.encode(nextCoding-lastX, eBuffer);
		      }
		      lastX=nextCoding;
		      nextCoding=cCache.putImageCodingLine[++nextCodingIndex];
		      if((nextCodingIndex & 1)!=0){
                        cCache.putImagePixel0Coder.encode(nextCoding-lastX, eBuffer);
		      }
		      else{
			cCache.putImagePixel1Coder.encode(nextCoding-lastX, eBuffer);
		      }
		      lastX=nextCoding;
		      nextCodingIndex++;
		    }
		  }
		}
	      }
	      int[] tmp=cCache.putImageReferenceLine;
	      cCache.putImageReferenceLine=cCache.putImageCodingLine;
	      cCache.putImageCodingLine=tmp;
	    }
          }
	  else{
	    // pixmap, not bitmap
            if(buff[s+21]==8){
              for(int i=24; i<length[0]; i++){
		eBuffer.encodeCachedValue(buff[nextSrc++], 8,
					  cCache.putImageByteCache, 4);
	      }
	    }
	    else{
              for(int i=24; i<length[0]; i++){
		eBuffer.encodeValue((int)buff[nextSrc++], 8);
              }
	    }
	  }
	}
	break;

      case 97: // X_QueryBestSize:
	{
	  eBuffer.encodeValue((int)buff[s+1], 2);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4),
				    29, cCache.drawableCache, 9);
	  eBuffer.encodeValue(rBuffer.getUINT(8), 16, 8);
	  eBuffer.encodeValue(rBuffer.getUINT(10), 16, 8);
	  sNumQueue.push(cCache.lastRequestSequenceNum, opcode);
	}
	break;
      case 91: // X_QueryColors:
	{
	  int numColors=((length[0]-8)>>>2);
	  eBuffer.encodeValue(numColors, 16, 5);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4), 29,
				    cCache.colormapCache, 9);
	  int nextSrc=8;
	  int predictedPixel=cCache.queryColorsLastPixel;
	  for(int i=0; i<numColors; i++){
	    int pixel=rBuffer.getULONG(nextSrc);
	    nextSrc+=4;
	    if(pixel==predictedPixel){
	      eBuffer.encodeValue(1, 1);
	    }
	    else{
	      eBuffer.encodeValue(0, 1);
	      eBuffer.encodeValue(pixel, 32, 9);
	    }
	    if(i==0){
	      cCache.queryColorsLastPixel=pixel;
	    }
	    predictedPixel=pixel+1;
	  }
	  sNumQueue.push(cCache.lastRequestSequenceNum, opcode);
	}
	break;
      case 98: // X_QueryExtension:
	{
	  int nameLength=rBuffer.getUINT(4);
	  eBuffer.encodeValue(nameLength, 16, 6);
	  int nextSrc=s+8;
	  for (; nameLength>0; nameLength--){
	    eBuffer.encodeValue((int)buff[nextSrc++], 8);
	  }
	  sNumQueue.push(cCache.lastRequestSequenceNum, opcode);
	}
	break;
      case 47: // X_QueryFont:
	{
	  int font=rBuffer.getULONG(4);
	  eBuffer.encodeValue(font-cCache.lastFont, 29, 5);
	  cCache.lastFont=font;
	  sNumQueue.push(cCache.lastRequestSequenceNum, opcode);
	}
	break;
      case 59: // X_SetClipRectangles:
	{
	  int numRectangles=((length[0]-12)>>>3);
	  eBuffer.encodeValue(numRectangles, 13, 4);
	  eBuffer.encodeValue((int)buff[s+1], 2);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4), 29,
				    cCache.gcCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getUINT(8), 16,
				    cCache.setClipRectanglesXCache, 8);
	  eBuffer.encodeCachedValue(rBuffer.getUINT(10), 16,
				    cCache.setClipRectanglesYCache, 8);
	  int nextSrc=12;
	  for(int i=0; i<numRectangles; i++){
	    for(int j=0; j < 4; j++){
	      eBuffer.encodeCachedValue(rBuffer.getUINT(nextSrc), 16,
					cCache.setClipRectanglesGeomCache[j], 8);
	      nextSrc+=2;
	    }
	  }
	}
	break;
      case 58: // X_SetDashes:
	{
	  int numDashes=rBuffer.getUINT(10);
	  eBuffer.encodeCachedValue(numDashes, 16,
				    cCache.setDashesLengthCache, 5);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4), 29,
				    cCache.gcCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getUINT(8), 16,
				    cCache.setDashesOffsetCache, 5);
	  int nextSrc=s+12;
	  for(int i=0; i<numDashes; i++){
	    eBuffer.encodeCachedValue(buff[nextSrc++], 8,
				      cCache.setDashesDashCache[i&1], 5);
	  }
	}
	break;
      case 22: X_SetSelectionOwner:
	{
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4), 29,
				    cCache.setSelectionOwnerCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(8), 29,
				    cCache.getSelectionOwnerSelectionCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(12), 32,
				    cCache.setSelectionOwnerTimestampCache, 9);
	}
	break;
      case 40: // X_TranslateCoords:
	{
	  eBuffer.encodeCachedValue(rBuffer.getULONG(4), 29,
				    cCache.translateCoordsSrcCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getULONG(8), 29,
				    cCache.translateCoordsDestCache, 9);
	  eBuffer.encodeCachedValue(rBuffer.getUINT(12), 16,
				    cCache.translateCoordsXCache, 8);
	  eBuffer.encodeCachedValue(rBuffer.getUINT(14), 16,
				    cCache.translateCoordsYCache, 8);
	  sNumQueue.push(cCache.lastRequestSequenceNum, opcode);
	}
	break;

      default:
	{
	  eBuffer.encodeValue((int) buff[s+1], 8);
	  eBuffer.encodeValue(rBuffer.getUINT(2), 16, 8);
	  int nextSrc=s+4;
	  for(int i=4; i<length[0]; i++){
	    eBuffer.encodeValue((int)buff[nextSrc++], 8);
	  }
	}
      }
//System.out.println("???");
      proxy.proxyWrite(id, eBuffer);
//System.out.println("????");
    }
//System.out.println("bye");
  }

  void putByte(byte[] array, int begin, int length) {
    try { clientOutput.write(array, begin, length); }
    catch (IOException e) {
    }
  }

  private int[] wvalue=new int[1];
  private byte[] opcode=new byte[1];
  private byte[] cValue=new byte[1];
  DecodeBuffer dBuffer=new DecodeBuffer(null, 0, 0);

  void doWrite(byte[] buff, int start, int length){
//System.out.println("doWrite:");
    byte[] message=null;
    int messageStart=0;
    int messageLength=0;

    wBuffer.reset();
    dBuffer.reset(buff, start, length);

    if(firstReply){
      dBuffer.decodeValue(wvalue, 8);
      int opcode=wvalue[0];
      dBuffer.decodeValue(wvalue, 8);
      int secondByte=wvalue[0];
      dBuffer.decodeValue(wvalue, 16);
      int major=wvalue[0];
      dBuffer.decodeValue(wvalue, 16);
      int minor=wvalue[0];
      dBuffer.decodeValue(wvalue, 16);
      int extraLength=wvalue[0];

      messageLength=8+(extraLength<<2);
      messageStart=wBuffer.addMessage(messageLength);
      message=wBuffer.getData();
      message[messageStart]=(byte)opcode;
      message[messageStart+1]=(byte)secondByte;
      wBuffer.putUINT(major, 2);
      wBuffer.putUINT(minor, 4);
      wBuffer.putUINT(extraLength, 6);
      int nextDest=messageStart+8;
      dBuffer.decodeValue(wvalue, 1);
      int cached=wvalue[0];
      if(cached!=0){
	System.arraycopy(ServerCache.lastInitReply.getData(), 0,
			 message, nextDest, messageLength-8);
      }
      else{
	for(int i=8; i<messageLength; i++){
	  dBuffer.decodeValue(wvalue, 8);
	  message[nextDest++]=(byte)wvalue[0];
	}
	ServerCache.lastInitReply.set(messageStart+8, messageLength-8, message);
      }
      imageByteOrder=message[messageStart+30];
//System.out.println("imageByteOrder="+imageByteOrder);
      bitmapBitOrder=message[messageStart+31];
//System.out.println("bitmapBitOrder="+bitmapBitOrder);
      scanlineUnit=message[messageStart+32];
//System.out.println("scanlineUnit="+scanlineUnit);
      scanlinePad=message[messageStart+32];
//System.out.println("scanlinePad="+scanlinePad);
      firstReply=false;
      putByte(wBuffer.getData(), 0, wBuffer.getLength());
      return;
    }

    while(dBuffer.decodeCachedValue(opcode, 8,
				    sCache.opcodeCache[sCache.lastOpcode], 8, true)){
      sCache.lastOpcode=(opcode[0]&0xff);

//      unsigned char *outputMessage = NULL;
//      unsigned int outputLength = 0;
//      unsigned int value;	// general-purpose temp variable for decoding ints

//      unsigned char cValue;	// general-purpose temp variable for decoding chars
//System.out.println("reply: opcode="+opcode[0]);
      int iopcode=(opcode[0]&0xff);
      if(iopcode==1){
	dBuffer.decodeCachedValue(wvalue, 16,
				  sCache.replySequenceNumCache, 7);
	int sequenceNumDiff=wvalue[0];
//System.out.println("sequenceNumDiff: "+sequenceNumDiff);
	int sequenceNum=sCache.lastSequenceNum+sequenceNumDiff;
	sequenceNum&=0xffff;
	sCache.lastSequenceNum=sequenceNum;
//System.out.println("sequenceNum: "+sequenceNum);
//	int nextSequenceNum;
//	int nextOpcode;
	int requestOpcode=256;
	if(sNumQueue.peek(nextSequenceNum, nextOpcode) &&
	   (nextSequenceNum[0]==sequenceNum)){
	  sNumQueue.pop(nextSequenceNum, nextOpcode,
			requestData[0], requestData[1],	requestData[2]);
	  requestOpcode=nextOpcode[0];
//System.out.println("  requestOpcode -> "+requestOpcode);
	  switch(requestOpcode){
	  case 84: // X_AllocColor:
	    {
	      messageLength=32;
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();
	      int nextDest=8;
	      for(int i=0; i<3; i++){
		dBuffer.decodeValue(wvalue, 1);
		if(wvalue[0]!=0){
		  wBuffer.putUINT(requestData[i][0], nextDest);
		}
		else{
		  dBuffer.decodeValue(wvalue, 16, 6);
		  wBuffer.putUINT(requestData[i][0]+wvalue[0], nextDest);
		}
		nextDest+=2;
	      }
	      dBuffer.decodeValue(wvalue, 32, 9);
	      wBuffer.putULONG(wvalue[0], 16);
	    }
	    break;
	  case 17: // X_GetAtomName:
	    {
	      dBuffer.decodeValue(wvalue, 16, 6);
	      int nameLength=wvalue[0];
	      messageLength=32+(nameLength+3)/4*4;
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();
	      wBuffer.putUINT(nameLength, 8);
	      int nextDest=messageStart+32;
	      cCache.internAtomTextCompressor.reset();
	      for(int i=0; i<nameLength; i++){
		message[nextDest++]=
		  cCache.internAtomTextCompressor.decodeChar(dBuffer);
	      }
	    }
	    break;
	  case 14: // X_GetGeometry:
	    {
	      messageLength=32;
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();

	      dBuffer.decodeCachedValue(cValue, 8, sCache.depthCache);
	      message[messageStart+1]=cValue[0];
	      dBuffer.decodeCachedValue(wvalue, 29,
					sCache.getGeometryRootCache, 9);
	      wBuffer.putULONG(wvalue[0], 8);
	      int nextDest=12;
	      for(int i=0; i<5; i++){
		dBuffer.decodeCachedValue(wvalue, 16,
					  sCache.getGeometryGeomCache[i], 8);
		wBuffer.putUINT(wvalue[0], nextDest);
		nextDest+=2;
	      }
	    }
	    break;
	  case 43: // X_GetInputFocus:
	    {
	      messageLength=32;
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();

	      dBuffer.decodeValue(wvalue, 2);
	      message[messageStart+1]=(byte)wvalue[0];
	      dBuffer.decodeCachedValue(wvalue, 29,
					sCache.getInputFocusWindowCache, 9);
	      wBuffer.putULONG(wvalue[0], 8);
	    }
	    break;
	  case 101: // X_GetKeyboardMapping:
	    {
	      dBuffer.decodeValue(wvalue, 1);
	      if(wvalue[0]!=0){
		int dataLength =
 		  ServerCache.getKeyboardMappingLastMap.getLength();
		messageLength=32+dataLength;
		messageStart=wBuffer.addMessage(messageLength);
		message=wBuffer.getData();
		message[messageStart+1]=
		  ServerCache.getKeyboardMappingLastKeysymsPerKeycode;
		System.arraycopy(ServerCache.getKeyboardMappingLastMap.getData(), 0,
				 message, messageStart+32, dataLength); 
		break;
	      }
	      dBuffer.decodeValue(wvalue, 8);
	      int numKeycodes=wvalue[0];
	      dBuffer.decodeValue(wvalue, 8, 4);
	      int keysymsPerKeycode=wvalue[0];
	      ServerCache.getKeyboardMappingLastKeysymsPerKeycode=(byte)keysymsPerKeycode;
	      messageLength=32+numKeycodes*keysymsPerKeycode*4;
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();
	      message[messageStart+1]=(byte)keysymsPerKeycode;
	      int nextDest=32;
	      int previous=0;
	      for(int count=numKeycodes*keysymsPerKeycode; count>0; --count){
		dBuffer.decodeValue(wvalue, 1);
		if(wvalue[0]!=0){
		  wBuffer.putULONG(0, nextDest);
		}
		else{
		  dBuffer.decodeCachedValue(wvalue, 24,
					    sCache.getKeyboardMappingKeysymCache, 9);
		  int keysym=wvalue[0];
		  dBuffer.decodeCachedValue(cValue, 8,
					    sCache.getKeyboardMappingLastByteCache, 5);
		  previous+=(int)(cValue[0]/*&0xff*/);
		  previous&=0xff;
		  wBuffer.putULONG(((keysym<<8)|previous), nextDest);
		}
		nextDest+=4;
	      }
	      ServerCache.getKeyboardMappingLastMap.set(messageStart+32,
							messageLength-32,
							message);
	    }
	    break;
	  case 119: // X_GetModifierMapping:
	    {
	      dBuffer.decodeValue(wvalue, 8);
	      int keycodesPerModifier=wvalue[0];
	      messageLength=32+(keycodesPerModifier<<3);
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();
	      message[messageStart+1]=(byte)keycodesPerModifier;
	      int nextDest=messageStart+32;
	      dBuffer.decodeValue(wvalue, 1);
	      if(wvalue[0]!=0){
		System.arraycopy(ServerCache.getModifierMappingLastMap.getData(), 0, 
				 message, messageStart+32, 
				 ServerCache.getModifierMappingLastMap.getLength());
		break;
	      }
	      for(int count=messageLength-32; count>0; count--){
		dBuffer.decodeValue(wvalue, 1);
		if(wvalue[0]!=0){
		  message[nextDest++] = 0;
		}
		else{
		  dBuffer.decodeValue(wvalue, 8);
		  message[nextDest++]=(byte)wvalue[0];
		}
	      }
	      ServerCache.getModifierMappingLastMap.set(messageStart+32,
							messageLength-32,
							message);
	    }
	    break;
	  case 20: //X_GetProperty:
	    {
	      dBuffer.decodeCachedValue(cValue, 8, sCache.getPropertyFormatCache);
	      byte format=(byte)cValue[0];
	      dBuffer.decodeValue(wvalue, 32, 9);
	      int len=wvalue[0];
	      int numBytes=len;
	      if(format==16){
		numBytes<<=1;
	      }
	      else if(format==32){
		numBytes<<=2;
	      }
	      messageLength=32+(numBytes+3)/4*4;
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();

	      message[messageStart+1]=format;
	      wBuffer.putULONG(len, 16);
	      dBuffer.decodeCachedValue(wvalue, 29,
					sCache.getPropertyTypeCache, 9);
	      wBuffer.putULONG(wvalue[0], 8);
	      dBuffer.decodeValue(wvalue, 32, 9);
	      wBuffer.putULONG(wvalue[0], 12);
	      int nextDest=messageStart+32;
	      if(format==8){
		if(requestData[0][0]==23/*XA_RESOURCE_MANAGER*/){
		  dBuffer.decodeValue(wvalue, 1);
		  if(wvalue[0]!=0){
		    System.arraycopy(ServerCache.xResources.getData(), 0,
				     message, nextDest,	     
				     ServerCache.xResources.getLength());
		    break;
		  }
		}
		sCache.getPropertyTextCompressor.reset();
		for(int i=0; i<numBytes; i++){
		  byte nextChar;
		  nextChar=message[nextDest++]=
		    sCache.getPropertyTextCompressor.decodeChar(dBuffer);
//System.out.print(new Character((char)nextChar));
		  if(nextChar==10){
		    sCache.getPropertyTextCompressor.reset(nextChar);
		  }
		}
		if(requestData[0][0]==23/*XA_RESOURCE_MANAGER*/){
		  ServerCache.xResources.set(messageStart+32, numBytes, message);
		}
	      }
	      else{
		for(int i=0; i<numBytes; i++){
		  dBuffer.decodeValue(wvalue, 8);
		  message[nextDest++]=(byte)wvalue[0];
		}
	      }
	    }
	    break;
	  case 23: // X_GetSelectionOwner:
	    {
	      messageLength=32;
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();

	      dBuffer.decodeCachedValue(wvalue, 29,
					sCache.getSelectionOwnerCache, 9);
	      wBuffer.putULONG(wvalue[0], 8);
	    }
	    break;
	  case 3: // X_GetWindowAttributes:
	    {
	      messageLength=44;
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();

	      dBuffer.decodeValue(wvalue, 2);
	      message[messageStart+1]=(byte)wvalue[0];
	      dBuffer.decodeCachedValue(wvalue, 29,
					sCache.visualCache, 9);
	      wBuffer.putULONG(wvalue[0], 8);
	      dBuffer.decodeCachedValue(wvalue, 16,
					sCache.getWindowAttributesClassCache, 3);
	      wBuffer.putUINT(wvalue[0], 12);
	      dBuffer.decodeCachedValue(cValue, 8,
					sCache.getWindowAttributesBitGravityCache);
	      message[messageStart+14]=(byte)cValue[0];
	      dBuffer.decodeCachedValue(cValue, 8,
					sCache.getWindowAttributesWinGravityCache);
	      message[messageStart+15]=(byte)cValue[0];
	      dBuffer.decodeCachedValue(wvalue, 32,
					sCache.getWindowAttributesPlanesCache, 9);
	      wBuffer.putULONG(wvalue[0], 16);
	      dBuffer.decodeCachedValue(wvalue, 32,
					sCache.getWindowAttributesPixelCache, 9);
	      wBuffer.putULONG(wvalue[0], 20);
	      dBuffer.decodeValue(wvalue, 1);
	      message[messageStart+24]=(byte)wvalue[0];
	      dBuffer.decodeValue(wvalue, 1);
	      message[messageStart+25]=(byte)wvalue[0];
	      dBuffer.decodeValue(wvalue, 2);
	      message[messageStart+26]=(byte)wvalue[0];
	      dBuffer.decodeValue(wvalue, 1);
	      message[messageStart+27]=(byte)wvalue[0];
	      dBuffer.decodeCachedValue(wvalue, 29,
					sCache.colormapCache, 9);
	      wBuffer.putULONG(wvalue[0], 28);
	      dBuffer.decodeCachedValue(wvalue, 32,
					sCache.getWindowAttributesAllEventsCache);
	      wBuffer.putULONG(wvalue[0], 32);
	      dBuffer.decodeCachedValue(wvalue, 32,
					sCache.getWindowAttributesYourEventsCache);
	      wBuffer.putULONG(wvalue[0], 36);
	      dBuffer.decodeCachedValue(wvalue, 16,
					sCache.getWindowAttributesDontPropagateCache);
	      wBuffer.putUINT(wvalue[0], 40);
	    }
	    break;
	  case 31: // X_GrabKeyboard:
	  case 26: // X_GrabPointer:
	    {
	      messageLength=32;
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();
	      dBuffer.decodeValue(wvalue, 3);
	      message[messageStart+1]=(byte)wvalue[0];
	    }
	    break;
	  case 16: // X_InternAtom:
	    {
	      messageLength=32;
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();
	      dBuffer.decodeValue(wvalue, 29, 9);
	      wBuffer.putULONG(wvalue[0], 8);
	    }
	    break;
	  case 99: // X_ListExtensions:
	    {
	      dBuffer.decodeValue(wvalue, 32, 8);
	      messageLength=32+(wvalue[0]<<2);
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();
	      dBuffer.decodeValue(wvalue, 8);
	      int numExtensions=wvalue[0];
	      message[messageStart+1]=(byte)numExtensions;
	      int nextDest=messageStart+32;
	      int len;
	      for(; numExtensions>0; numExtensions--){
		dBuffer.decodeValue(wvalue, 8);
		len=wvalue[0];
		message[nextDest++]=(byte)len;
		for(; length>0; length--){
		  dBuffer.decodeValue(wvalue, 8);
		  message[nextDest++]=(byte)wvalue[0];
		}
	      }
	    }
	    break;
	  case 49: // X_ListFonts:
	    {
	      dBuffer.decodeValue(wvalue, 32, 8);
	      messageLength=32+(wvalue[0]<<2);
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();
	      dBuffer.decodeValue(wvalue, 16, 6);
	      int numFonts=wvalue[0];
	      wBuffer.putUINT(numFonts, 8);
	      int nextDest=messageStart+32;
	      for (; numFonts>0; numFonts--){
		dBuffer.decodeValue(wvalue, 8);
		int len=wvalue[0];
		message[nextDest++]=(byte)len;
		sCache.getPropertyTextCompressor.reset();
		for(; len>0; len--){
		  message[nextDest++]=
		    sCache.getPropertyTextCompressor.decodeChar(dBuffer);
		}
	      }
	    }
	    break;
	  case 92: // X_LookupColor:
	  case 85: // X_AllocNamedColor:
	    {
	      messageLength=32;
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();

	      int nextDest=8;
	      if(requestOpcode==85/*X_AllocNamedColor*/){
		dBuffer.decodeValue(wvalue, 32, 9);
		wBuffer.putULONG(wvalue[0], nextDest);
		nextDest+=4;
	      }
	      int count=3;
	      do{
		dBuffer.decodeValue(wvalue, 16, 9);
		int val=wvalue[0];
		wBuffer.putUINT(val, nextDest);
		dBuffer.decodeValue(wvalue, 16, 5);
		int visualColor=wvalue[0];
		visualColor+=val;
		visualColor&=0xffff;
		wBuffer.putUINT(visualColor, nextDest+6);
		nextDest+=2;
	      }
	      while(--count>0);
	    }
	    break;
	  case 97: // X_QueryBestSize:
	    {
	      messageLength=32;
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();
	      dBuffer.decodeValue(wvalue, 16, 8);
	      wBuffer.putUINT(wvalue[0], 8);
	      dBuffer.decodeValue(wvalue, 16, 8);
	      wBuffer.putUINT(wvalue[0], 10);
	    }
	    break;
	  case 91: // X_QueryColors:
	    {
	      dBuffer.decodeValue(wvalue, 1, 1);
	      int cached=wvalue[0];
	      if(cached!=0){
		int numColors=sCache.queryColorsLastReply.getLength()/6;
		messageLength= 32+(numColors<<3);
		messageStart=wBuffer.addMessage(messageLength);
		message=wBuffer.getData();
		wBuffer.putUINT(numColors, 8);
		byte[] tmp=sCache.queryColorsLastReply.getData();
		int nextSrc=0;
		int nextDest=messageStart+32;
		for(; numColors>0; numColors--){
		  for(int i=0; i<6; i++){
		    message[nextDest++]=tmp[nextSrc++];
		  }
		  nextDest+=2;
		}
	      }
	      else{
		dBuffer.decodeValue(wvalue, 16, 5);
		int numColors=wvalue[0];
		messageLength=32+(numColors<<3);
		messageStart=wBuffer.addMessage(messageLength);
		message=wBuffer.getData();
		wBuffer.putUINT(numColors, 8);
		int nextDest=32;
		for(int c=0; c<numColors; c++){
		  for(int i=0; i<3; i++){
		    dBuffer.decodeValue(wvalue, 16);
		    wBuffer.putUINT(wvalue[0], nextDest);
		    nextDest+=2;
		  }
		}
		sCache.queryColorsLastReply.set(messageStart+32,
						numColors*6,
						message);
		int nextSrc=messageStart+nextDest-1;
		nextDest=messageStart+32+((numColors-1)<<3)+5;
		for(; numColors>1; numColors--){
		  for(int i=0; i<6; i++){
		    message[nextDest--]=message[nextSrc--];
		  }
		  nextDest-=2;
		}
	      }
	    }
	    break;
	  case 98: // X_QueryExtension:
	    {
	      messageLength = 32;
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();
	      dBuffer.decodeValue(wvalue, 1);
	      message[messageStart+8]=(byte)wvalue[0];
	      dBuffer.decodeValue(wvalue, 8);
	      message[messageStart+9]=(byte)wvalue[0];
	      dBuffer.decodeValue(wvalue, 8);
	      message[messageStart+10]=(byte)wvalue[0];
	      dBuffer.decodeValue(wvalue, 8);
	      message[messageStart+11]=(byte)wvalue[0];
	    }
	    break;
	  case 47: // X_QueryFont:
	    {

	      dBuffer.decodeValue(wvalue, 16, 8);
	      int numProperties=wvalue[0];
	      dBuffer.decodeValue(wvalue, 32, 10);
	      int numCharInfos=wvalue[0];

	      messageLength=60+numProperties*8+numCharInfos*12;
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();
	      wBuffer.putUINT(numProperties, 46);
	      wBuffer.putULONG(numCharInfos, 56);
	      decodeCharInfo(dBuffer, 8, wBuffer); 
	      decodeCharInfo(dBuffer, 24, wBuffer);
	      dBuffer.decodeValue(wvalue, 16, 9);
	      wBuffer.putUINT(wvalue[0], 40);
	      dBuffer.decodeValue(wvalue, 16, 9);
	      wBuffer.putUINT(wvalue[0], 42);
	      dBuffer.decodeValue(wvalue, 16, 9);
	      wBuffer.putUINT(wvalue[0], 44);
	      dBuffer.decodeValue(wvalue, 1);
	      message[messageStart+48]=(byte)wvalue[0];
	      dBuffer.decodeValue(wvalue, 8);
	      message[messageStart+49]=(byte)wvalue[0];
	      dBuffer.decodeValue(wvalue, 8);
	      message[messageStart+50]=(byte)wvalue[0];
	      dBuffer.decodeValue(wvalue, 1);
	      message[messageStart+51]=(byte)wvalue[0];
	      dBuffer.decodeValue(wvalue, 16, 9);
	      wBuffer.putUINT(wvalue[0], 52);
	      dBuffer.decodeValue(wvalue, 16, 9);
	      wBuffer.putUINT(wvalue[0], 54);
	      int nextDest=60;
	      dBuffer.decodeValue(wvalue, 1);
	      if(wvalue[0]!=0){
		nextDest+=messageStart;
		dBuffer.decodeValue(wvalue, 4);
		int index=wvalue[0];
		byte[][] tmp=new byte[1][];
		ServerCache.queryFontFontCache.get(index, wvalue, tmp);
		int len=wvalue[0];
		System.arraycopy(tmp[0], 0, message, nextDest, wvalue[0]);
		break;
	      }
	      int saveDest = nextDest;
	      int len=numProperties*8+numCharInfos*12;
	      for(; numProperties>0; numProperties--){
		dBuffer.decodeValue(wvalue, 32, 9);
		wBuffer.putULONG(wvalue[0], nextDest);
		dBuffer.decodeValue(wvalue, 32, 9);
		wBuffer.putULONG(wvalue[0], nextDest+4);
		nextDest+=8;
	      }
	      for(; numCharInfos>0; numCharInfos--){
		decodeCharInfo(dBuffer, nextDest, wBuffer);
		nextDest+=12;
	      }
	      ServerCache.queryFontFontCache.set(messageStart+saveDest, 
						 len,
						 message
						 );
	    }
	    break;
	  case 38: // X_QueryPointer:
	    {
	      messageLength = 32;
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();

	      dBuffer.decodeValue(wvalue, 1);
	      message[messageStart+1]=(byte)wvalue[0];
	      dBuffer.decodeCachedValue(wvalue, 29,
					sCache.queryPointerRootCache, 9);
	      wBuffer.putULONG(wvalue[0], 8);
	      dBuffer.decodeCachedValue(wvalue, 29,
					sCache.queryPointerChildCache, 9);
	      wBuffer.putULONG(wvalue[0], 12);
	      dBuffer.decodeCachedValue(wvalue, 16,
					sCache.motionNotifyRootXCache, 8);
	      sCache.motionNotifyLastRootX+=wvalue[0];
	      wBuffer.putUINT(sCache.motionNotifyLastRootX, 16);
	      dBuffer.decodeCachedValue(wvalue, 16,
					sCache.motionNotifyRootYCache, 8);
	      sCache.motionNotifyLastRootY+=wvalue[0];
	      wBuffer.putUINT(sCache.motionNotifyLastRootY, 18);
	      dBuffer.decodeCachedValue(wvalue, 16,
					sCache.motionNotifyEventXCache, 8);
	      wBuffer.putUINT(sCache.motionNotifyLastRootX+wvalue[0], 20);
	      dBuffer.decodeCachedValue(wvalue, 16,
					sCache.motionNotifyEventYCache, 8);
	      wBuffer.putUINT(sCache.motionNotifyLastRootY+wvalue[0], 22);
	      dBuffer.decodeCachedValue(wvalue, 16,
					sCache.motionNotifyStateCache);
	      wBuffer.putUINT(wvalue[0], 24);
	    }
	    break;
	  case 15: // X_QueryTree:
	    {
	      dBuffer.decodeValue(wvalue, 8);
	      int secondByte=wvalue[0];
	      dBuffer.decodeValue(wvalue, 32);
	      int replyLength=wvalue[0];
	      messageLength=32+(replyLength<<2);
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();
	      message[messageStart+1]=(byte)secondByte;
	      int nextDest=messageStart+8;
	      for(int i=8; i<messageLength; i++){
		dBuffer.decodeValue(wvalue, 8);
		message[nextDest++]=(byte)wvalue[0];
	      }
	    }
	    break;
	  case 40: // X_TranslateCoords:
	    {
	      messageLength=32;
	      messageStart=wBuffer.addMessage(messageLength);
	      message=wBuffer.getData();

	      dBuffer.decodeValue(wvalue, 1);
	      message[messageStart+1]=(byte)wvalue[0];
	      dBuffer.decodeCachedValue(wvalue, 29,
					sCache.translateCoordsChildCache, 9);
	      wBuffer.putULONG(wvalue[0], 8);
	      dBuffer.decodeCachedValue(wvalue, 16,
					sCache.translateCoordsXCache, 8);
	      wBuffer.putUINT(wvalue[0], 12);
	      dBuffer.decodeCachedValue(wvalue, 16,
					sCache.translateCoordsYCache, 8);
	      wBuffer.putUINT(wvalue[0], 14);
	    }
	    break;
	  default:
	    {
//	      System.err.println("assertion failed in ClientProxyReader::processMessage():");
//	      System.err.println(" no matching request for reply with sequence number "+sequenceNum);
	    }
	  }
	}
	else{
	  dBuffer.decodeValue(wvalue, 8);
	  int secondByte=wvalue[0];
	  dBuffer.decodeValue(wvalue, 32);
	  int replyLength=wvalue[0];

	  messageLength=32+(replyLength<<2);
	  messageStart=wBuffer.addMessage(messageLength);
	  message=wBuffer.getData();

	  message[messageStart+1]=(byte)secondByte;
	  int nextDest=messageStart+8;
	  for(int i=8; i<messageLength; i++){
	    dBuffer.decodeValue(wvalue, 8);
	    message[nextDest++]=(byte)wvalue[0];
	  }
	}
	wBuffer.putULONG((messageLength-32)>>>2, 4);	
      }
      else{
	// event or error
	dBuffer.decodeCachedValue(wvalue, 16,
				  sCache.eventSequenceNumCache, 7);
	int sequenceNumDiff=wvalue[0];
	sCache.lastSequenceNum+=sequenceNumDiff;
	sCache.lastSequenceNum&=0xffff;

	messageLength=32;
	messageStart=wBuffer.addMessage(messageLength);
	message=wBuffer.getData();

	// check if this is an error that matches a sequence number for
	// which we were expecting a reply
//	unsigned short int dummySequenceNum;
//	unsigned char dummyOpcode;
	if (sNumQueue.peek(dummySequenceNum, dummyOpcode) &&
	  ((int)dummySequenceNum[0]==sCache.lastSequenceNum))
	  sNumQueue.pop(dummySequenceNum, dummyOpcode);

	switch(iopcode){
	case 0:
	  {
	    dBuffer.decodeCachedValue(cValue, 8, sCache.errorCodeCache);
	    byte code=cValue[0];
	    message[messageStart+1]=code;
	    if((code!=11)&&(code!=8)&&(code!=15)&&(code != 1)){
	      dBuffer.decodeValue(wvalue, 32, 16);
	      wBuffer.putULONG(wvalue[0], 4);
	    }
	    if(code>=18){
	      dBuffer.decodeCachedValue(wvalue, 16, sCache.errorMinorCache);
	      wBuffer.putUINT(wvalue[0], 8);
	    }
	    dBuffer.decodeCachedValue(cValue, 8, sCache.errorMajorCache);
	    message[messageStart+10]=cValue[0];
	    if(code>=18){
	      int nextDest=messageStart+11;
	      for(int i=11; i<32; i++){
		dBuffer.decodeValue(wvalue, 8);
		message[nextDest++]=(byte)wvalue[0];
//bug????????	message[nextDest++]=(byte)cValue[0];
	      }
	    }
	  }
	  break;
	case 4: // ButtonPress
	case 5: // ButtonRelease
	case 2: // KeyPress
	case 3: // KeyRelease
	case 6: // MotionNotify
	case 7: // EnterNotify
	case 8: // LeaveNotify
	  {
	    if(iopcode==6/*MotionNotify*/){
	      dBuffer.decodeValue(wvalue, 1);
	    }
	    else if((iopcode==7/*EnterNotify*/)||
		    (iopcode==8/*LeaveNotify*/)){
	      dBuffer.decodeValue(wvalue, 3);
	    }
	    else if(iopcode==3/*KeyRelease*/){
	      dBuffer.decodeValue(wvalue, 1);
	      if(wvalue[0]!=0){
		wvalue[0]=sCache.keyPressLastKey;
	      }
	      else{
		dBuffer.decodeValue(wvalue, 8);
	      }
	    }
	    else if((iopcode==4/*ButtonPress*/)||
		    (iopcode==5/*ButtonRelease*/)){
	      dBuffer.decodeCachedValue(cValue, 8,
					sCache.buttonCache);
	      wvalue[0]=(int)cValue[0];
	    }
	    else{
	      dBuffer.decodeValue(wvalue, 8);
	    }
	    message[messageStart+1]=(byte)wvalue[0];
	    dBuffer.decodeCachedValue(wvalue, 32,
				      sCache.motionNotifyTimestampCache, 9);
	    sCache.lastTimestamp+=wvalue[0];
	    wBuffer.putULONG(sCache.lastTimestamp, 4);
	    int nextDest=8;
	    boolean skipRest=false;
	    if(iopcode==3/*KeyRelease*/){
	      dBuffer.decodeValue(wvalue, 1);
	      if(wvalue[0]!=0){
		nextDest+=messageStart;
		for(int i=0; i<23; i++){
		  message[nextDest++]=sCache.keyPressCache[i];
		}
		skipRest=true;
	      }
	    }
	    if(!skipRest){
	      for(int i=0; i<3; i++){
		dBuffer.decodeCachedValue(wvalue, 29,
					  sCache.motionNotifyWindowCache[i], 6);
		wBuffer.putULONG(wvalue[0], nextDest);
		nextDest+=4;
	      }
	      dBuffer.decodeCachedValue(wvalue, 16,
					sCache.motionNotifyRootXCache, 6);
	      sCache.motionNotifyLastRootX+=wvalue[0];
	      wBuffer.putUINT(sCache.motionNotifyLastRootX, 20);
	      dBuffer.decodeCachedValue(wvalue, 16,
					sCache.motionNotifyRootYCache, 6);
	      sCache.motionNotifyLastRootY+=wvalue[0];
	      wBuffer.putUINT(sCache.motionNotifyLastRootY, 22);
	      dBuffer.decodeCachedValue(wvalue, 16,
					sCache.motionNotifyEventXCache, 6);
	      wBuffer.putUINT(sCache.motionNotifyLastRootX+wvalue[0], 24);
	      dBuffer.decodeCachedValue(wvalue, 16,
					sCache.motionNotifyEventYCache, 6);
	      wBuffer.putUINT(sCache.motionNotifyLastRootY+wvalue[0], 26);
	      dBuffer.decodeCachedValue(wvalue, 16,
					sCache.motionNotifyStateCache);
	      wBuffer.putUINT(wvalue[0], 28);
	      if((iopcode==7/*EnterNotify*/)||
		 (iopcode==8/*LeaveNotify*/)){
		dBuffer.decodeValue(wvalue, 2);
	      }
	      else{
		dBuffer.decodeValue(wvalue, 1);
	      }
	      message[messageStart+30]=(byte)wvalue[0];
	      if((iopcode==7/*EnterNotify*/)||
		 (iopcode==8/*LeaveNotify*/)){
		dBuffer.decodeValue(wvalue, 2);
		message[messageStart+31]=(byte)wvalue[0];
	      }
	      else if(iopcode==2/*KeyPress*/){
		sCache.keyPressLastKey=message[messageStart+1];
		for(int i=8; i<31; i++){
		  sCache.keyPressCache[i-8]=message[messageStart+i];
		}
	      }
	    }
	  }
	  break;
	case 32: // ColormapNotify:
	  {
	    dBuffer.decodeCachedValue(wvalue, 29,
				      sCache.colormapNotifyWindowCache, 8);
	    wBuffer.putULONG(wvalue[0], 4);
	    dBuffer.decodeCachedValue(wvalue, 29,
				      sCache.colormapNotifyColormapCache, 8);
	    wBuffer.putULONG(wvalue[0], 8);
	    dBuffer.decodeValue(wvalue, 1);
	    message[messageStart+12]=(byte)wvalue[0];
	    dBuffer.decodeValue(wvalue, 1);
	    message[messageStart+13]=(byte)wvalue[0];
	  }
	  break;
	case 22: // ConfigureNotify:
	  {
	    int nextDest=/*messageStart*/+4;
	    for(int i=0; i<3; i++){
	      dBuffer.decodeCachedValue(wvalue, 29,
					sCache.configureNotifyWindowCache[i], 9);
	      wBuffer.putULONG(wvalue[0], nextDest);
	      nextDest+=4;
	    }
	    for(int j=0; j<5; j++){
	      dBuffer.decodeCachedValue(wvalue, 16,
					sCache.configureNotifyGeomCache[j], 8);
	      wBuffer.putUINT(wvalue[0], nextDest);
	      nextDest+=2;
	    }
	    dBuffer.decodeValue(wvalue, 1);
	    message[messageStart+nextDest]=(byte)wvalue[0];
	  }
	  break;
	case 16: // CreateNotify:
	  {
	    dBuffer.decodeCachedValue(wvalue, 29,
				      sCache.createNotifyWindowCache, 9);
	    wBuffer.putULONG(wvalue[0], 4);
	    dBuffer.decodeValue(wvalue, 29, 5);
	    sCache.createNotifyLastWindow+=wvalue[0];
	    sCache.createNotifyLastWindow&=0x1fffffff;
	    wBuffer.putULONG(sCache.createNotifyLastWindow, 8);
	    int nextDest=12;
	    for(int i=0; i<5; i++){
	      dBuffer.decodeValue(wvalue, 16, 9);
	      wBuffer.putUINT(wvalue[0], nextDest);
	      nextDest+=2;
	    }
	    dBuffer.decodeValue(wvalue, 1);
	    message[messageStart+nextDest]=(byte)wvalue[0];
	  }
	  break;
	case 12: // Expose:
	  {
	    dBuffer.decodeCachedValue(wvalue, 29,
				      sCache.exposeWindowCache, 9);
	    wBuffer.putULONG(wvalue[0], 4);
	    int nextDest=8;
	    for(int i=0; i<5; i++){
	      dBuffer.decodeCachedValue(wvalue, 16,
					sCache.exposeGeomCache[i], 6);
	      wBuffer.putUINT(wvalue[0], nextDest);
	      nextDest+=2;
	    }
	  }
	  break;
	case  9: // FocusIn:
	case 10: //FocusOut:
	  {
	    dBuffer.decodeValue(wvalue, 3);
	    message[messageStart+1]=(byte)wvalue[0];
	    dBuffer.decodeCachedValue(wvalue, 29,
				      sCache.focusInWindowCache, 9);
	    wBuffer.putULONG(wvalue[0], 4);
	    dBuffer.decodeValue(wvalue, 2);
	    message[messageStart+8]=(byte)wvalue[0];
	  }
	  break;
	case 11: // KeymapNotify:
	  {
	    dBuffer.decodeValue(wvalue, 1);
	    if(wvalue[0]!=0){
	      System.arraycopy(ServerCache.lastKeymap.getData(), 0,
			       message, messageStart+1,
			       31);
	    }
	    else{
	      int nextDest=messageStart+1;
	      for(int i=1; i<32; i++){
		dBuffer.decodeValue(wvalue, 8);
		message[nextDest++]=(byte)wvalue[0];
	      }
	      ServerCache.lastKeymap.set(messageStart+1, 31, message);
	    }
	  }
	  break;
	case 19: // MapNotify:
	case 18: // UnmapNotify:
	case 17: // DestroyNotify:
	  {
	    dBuffer.decodeCachedValue(wvalue, 29,
				      sCache.mapNotifyEventCache, 9);
	    wBuffer.putULONG(wvalue[0], 4);
	    dBuffer.decodeCachedValue(wvalue, 29,
				      sCache.mapNotifyWindowCache, 9);
	    wBuffer.putULONG(wvalue[0], 8);
	    if ((iopcode==19/*MapNotify*/)||
		(iopcode==18/*UnmapNotify*/)){
	      dBuffer.decodeValue(wvalue, 1);
	      message[messageStart+12]=(byte)wvalue[0];
	    }
	  }
	  break;
	case 14: // NoExpose:
	  {
	    dBuffer.decodeCachedValue(wvalue, 29,
				      sCache.noExposeDrawableCache, 9);
	    wBuffer.putULONG(wvalue[0], 4);
	    dBuffer.decodeCachedValue(wvalue, 16,
				      sCache.noExposeMinorCache);
	    wBuffer.putULONG(wvalue[0], 8);
	    dBuffer.decodeCachedValue(cValue, 8,
				      sCache.noExposeMajorCache);
	    message[messageStart+10]=cValue[0];
	  }
	  break;
	case 28: // PropertyNotify:
	  {
	    dBuffer.decodeCachedValue(wvalue, 29,
				      sCache.propertyNotifyWindowCache, 9);
	    wBuffer.putULONG(wvalue[0], 4);
	    dBuffer.decodeCachedValue(wvalue, 29,
				      sCache.propertyNotifyAtomCache, 9);
	    wBuffer.putULONG(wvalue[0], 8);
	    dBuffer.decodeValue(wvalue, 32, 9);
	    sCache.lastTimestamp+=wvalue[0];
	    wBuffer.putULONG(sCache.lastTimestamp, 12);
	    dBuffer.decodeValue(wvalue, 1);
	    message[messageStart+16]=(byte)wvalue[0];
	  }
	  break;
	case 21: // ReparentNotify:
	  {
	    int nextDest=4;
	    for(int i=0; i<3; i++){
	      dBuffer.decodeCachedValue(wvalue, 29,
					sCache.reparentNotifyWindowCache, 9);
	      wBuffer.putULONG(wvalue[0], nextDest);
	      nextDest+=4;
	    }
	    dBuffer.decodeValue(wvalue, 16, 6);
	    wBuffer.putUINT(wvalue[0], nextDest);
	    dBuffer.decodeValue(wvalue, 16, 6);
	    wBuffer.putUINT(wvalue[0], nextDest+2);
	    dBuffer.decodeValue(wvalue, 1);
	    message[messageStart+20]=(byte)wvalue[0];
	  }
	  break;
	case 29: // SelectionClear:
	  {
	    dBuffer.decodeValue(wvalue, 32, 9);
	    sCache.lastTimestamp+=wvalue[0];
	    wBuffer.putULONG(sCache.lastTimestamp, 4);
	    dBuffer.decodeCachedValue(wvalue, 29,
				      sCache.selectionClearWindowCache, 9);
	    wBuffer.putULONG(wvalue[0], 8);
	    dBuffer.decodeCachedValue(wvalue, 29,
				      sCache.selectionClearAtomCache, 9);
	    wBuffer.putULONG(wvalue[0], 12);
	  }
	  break;
	case 30: // SelectionRequest:
	  {
	    dBuffer.decodeValue(wvalue, 32, 9);
	    sCache.lastTimestamp+=wvalue[0];
	    wBuffer.putULONG(sCache.lastTimestamp, 4);
	    dBuffer.decodeCachedValue(wvalue, 29,
				      sCache.selectionClearWindowCache, 9);
	    wBuffer.putULONG(wvalue[0], 8);
	    dBuffer.decodeCachedValue(wvalue, 29,
				      sCache.selectionClearWindowCache, 9);
	    wBuffer.putULONG(wvalue[0], 12);
	    dBuffer.decodeCachedValue(wvalue, 29,
				      sCache.selectionClearAtomCache, 9);
	    wBuffer.putULONG(wvalue[0], 16);
	    dBuffer.decodeCachedValue(wvalue, 29,
				      sCache.selectionClearAtomCache, 9);
	    wBuffer.putULONG(wvalue[0], 20);
	    dBuffer.decodeCachedValue(wvalue, 29,
				      sCache.selectionClearAtomCache, 9);
	    wBuffer.putULONG(wvalue[0], 24);
	  }
	  break;
	case 15: // VisibilityNotify:
	  {
	    dBuffer.decodeCachedValue(wvalue, 29,
				      sCache.visibilityNotifyWindowCache, 9);
	    wBuffer.putULONG(wvalue[0], 4);
	    dBuffer.decodeValue(wvalue, 2);
	    message[messageStart+8]=(byte)wvalue[0];
	  }
	  break;
	default:
	  {
	    dBuffer.decodeValue(wvalue, 8);
	    int secondByte=wvalue[0];
	    message[messageStart+1]=(byte)secondByte;
	    int nextDest=messageStart+4;
	    for(int i=4; i<messageLength; i++){
	      dBuffer.decodeValue(wvalue, 8);
	      message[nextDest++]=(byte)wvalue[0];
	    }
	  }
	}
      }
      message[messageStart]=opcode[0];
      wBuffer.putUINT(sCache.lastSequenceNum, 2);
      putByte(message, messageStart, messageLength);
    }
  }

  void decodeCharInfo(DecodeBuffer dBuffer, int nextDest, WriteBuffer wBuffer){
    dBuffer.decodeCachedValue(wvalue, 32,
			      sCache.queryFontCharInfoCache[0], 6);
    wBuffer.putUINT(wvalue[0]/*&0xffff*/, nextDest);
    wBuffer.putUINT(wvalue[0]>>>16, nextDest+10);
    nextDest+=2;
    for(int i=1; i<5; i++){
      dBuffer.decodeCachedValue(wvalue, 16,
				sCache.queryFontCharInfoCache[i], 6);
      wBuffer.putUINT(wvalue[0], nextDest);
      nextDest+=2;
    }
  }
}
